/* 
 * path_driver.c -- drive an editor across a set of paths
 * 
 * ====================================================================
 * Copyright (c) 2000-2004 CollabNet.  All rights reserved.
 *
 * This software is licensed as described in the file COPYING, which
 * you should have received as part of this distribution.  The terms
 * are also available at http://subversion.tigris.org/license-1.html.
 * If newer versions of this license are posted there, you may use a
 * newer version instead, at your option.
 *
 * This software consists of voluntary contributions made by many
 * individuals.  For exact contribution history, see the revision
 * history and logs, available at http://subversion.tigris.org/.
 * ====================================================================
 */


#include <assert.h>
#include <apr_pools.h>
#include <apr_strings.h>

#include "svn_types.h"
#include "svn_delta.h"
#include "svn_pools.h"
#include "svn_path.h"


/*** Helper functions. ***/

typedef struct dir_stack_t
{
  void *dir_baton;   /* the dir baton. */
  apr_pool_t *pool;  /* the pool associated with the dir baton. */

} dir_stack_t;


/* Call EDITOR's open_directory() function with the PATH and REVISION
 * arguments, and then add the resulting dir baton to the dir baton
 * stack. 
 */
static svn_error_t *
open_dir (apr_array_header_t *db_stack,
          const svn_delta_editor_t *editor,
          const char *path,
          svn_revnum_t revision,
          apr_pool_t *pool)
{
  void *parent_db, *db;
  dir_stack_t *item;
  apr_pool_t *subpool;

  /* Assert that we are in a stable state. */
  assert (db_stack && db_stack->nelts);

  /* Get the parent dir baton. */
  item = APR_ARRAY_IDX (db_stack, db_stack->nelts - 1, void *);
  parent_db = item->dir_baton;

  /* Call the EDITOR's open_directory function to get a new directory
     baton. */
  subpool = svn_pool_create (pool);
  SVN_ERR (editor->open_directory (path, parent_db, revision, subpool, &db));

  /* Now add the dir baton to the stack. */
  item = apr_pcalloc (subpool, sizeof (*item));
  item->dir_baton = db;
  item->pool = subpool;
  APR_ARRAY_PUSH (db_stack, dir_stack_t *) = item;

  return SVN_NO_ERROR;
}


/* Pop a directory from the dir baton stack and update the stack
 * pointer.
 *
 * This function calls the EDITOR's close_directory() function.
 */
static svn_error_t *
pop_stack (apr_array_header_t *db_stack,
           const svn_delta_editor_t *editor)
{
  dir_stack_t *item;

  /* Assert that we are in a stable state. */
  assert (db_stack && db_stack->nelts);

  /* Close the most recent directory pushed to the stack. */
  item = APR_ARRAY_IDX (db_stack, db_stack->nelts - 1, dir_stack_t *);
  (void) apr_array_pop (db_stack);
  SVN_ERR (editor->close_directory (item->dir_baton, item->pool));
  svn_pool_destroy (item->pool);

  return SVN_NO_ERROR;
}


/* Count the number of path components in PATH. */
static int
count_components (const char *path)
{
  int count = 1;
  const char *instance = path;

  if ((strlen (path) == 1) && (path[0] == '/'))
    return 0;

  do
    {
      instance++;
      instance = strchr (instance, '/');
      if (instance)
        count++;
    }
  while (instance);

  return count;
}


/* qsort-ready comparison function. */
static int compare_paths (const void *a, const void *b)
{
  const char *item1 = *((const char * const *) a);
  const char *item2 = *((const char * const *) b);
  return svn_path_compare_paths (item1, item2);
}



/*** Public interfaces ***/
svn_error_t *
svn_delta_path_driver (const svn_delta_editor_t *editor,
                       void *edit_baton,
                       svn_revnum_t revision,
                       apr_array_header_t *paths,
                       svn_delta_path_driver_cb_func_t callback_func,
                       void *callback_baton,
                       apr_pool_t *pool)
{
  apr_array_header_t *db_stack = apr_array_make (pool, 4, sizeof (void *));
  const char *last_path = NULL;
  int i = 0;
  void *parent_db = NULL, *db = NULL;
  const char *path;
  apr_pool_t *subpool = svn_pool_create (pool);
  apr_pool_t *iterpool = svn_pool_create (pool);
  dir_stack_t *item = apr_pcalloc (subpool, sizeof (*item));

  /* Do nothing if there are no paths. */
  if (! paths->nelts)
    return SVN_NO_ERROR;

  /* Sort the paths in a depth-first directory-ish order. */
  qsort (paths->elts, paths->nelts, paths->elt_size, compare_paths);

  /* If the root of the edit is also a target path, we want to call
     the callback function to let the user open the root directory and
     do what needs to be done.  Otherwise, we'll do the open_root()
     ourselves. */
  path = APR_ARRAY_IDX (paths, 0, const char *);
  if (svn_path_is_empty (path))
    {
      SVN_ERR (callback_func (&db, NULL, callback_baton, path, subpool));
      last_path = path;
      i++;
    }
  else
    {
      SVN_ERR (editor->open_root (edit_baton, revision, subpool, &db));
    }
  item->pool = subpool;
  item->dir_baton = db;
  APR_ARRAY_PUSH (db_stack, void *) = item;

  /* Now, loop over the commit items, traversing the URL tree and
     driving the editor. */
  for (; i < paths->nelts; i++)
    {
      const char *pdir, *bname;
      const char *common = "";
      size_t common_len;

      /* Clear the iteration pool. */
      svn_pool_clear (iterpool);

      /* Get the next path. */
      path = APR_ARRAY_IDX (paths, i, const char *);

      /*** Step A - Find the common ancestor of the last path and the
           current one.  For the first iteration, this is just the
           empty string. ***/
      if (i > 0)
        common = svn_path_get_longest_ancestor (last_path, path, iterpool);
      common_len = strlen (common);

      /*** Step B - Close any directories between the last path and
           the new common ancestor, if any need to be closed.
           Sometimes there is nothing to do here (like, for the first
           iteration, or when the last path was an ancestor of the
           current one). ***/
      if ((i > 0) && (strlen (last_path) > common_len))
        {
          const char *rel = last_path + (common_len ? (common_len + 1) : 0);
          int count = count_components (rel);
          while (count--)
            {
              SVN_ERR (pop_stack (db_stack, editor));
            }
        }

      /*** Step C - Open any directories between the common ancestor
           and the parent of the current path. ***/
      svn_path_split (path, &pdir, &bname, iterpool);
      if (strlen (pdir) > common_len)
        {
          const char *piece = pdir + common_len + 1;

          while (1)
            {
              const char *rel = pdir;

              /* Find the first separator. */
              piece = strchr (piece, '/');

              /* Calculate REL as the portion of PDIR up to (but not
                 including) the location to which PIECE is pointing. */
              if (piece)
                rel = apr_pstrmemdup (iterpool, pdir, piece - pdir);

              /* Open the subdirectory. */
              SVN_ERR (open_dir (db_stack, editor, rel, revision, pool));
              
              /* If we found a '/', advance our PIECE pointer to
                 character just after that '/'.  Otherwise, we're
                 done.  */
              if (piece)
                piece++;    
              else
                break;
            }
        }

      /*** Step D - Tell our caller to handle the current path. ***/
      item = APR_ARRAY_IDX (db_stack, db_stack->nelts - 1, void *);
      parent_db = item->dir_baton;
      subpool = svn_pool_create (pool);
      SVN_ERR (callback_func (&db, parent_db, callback_baton, path, subpool));
      if (db)
        {
          item = apr_pcalloc (subpool, sizeof (*item));
          item->dir_baton = db;
          item->pool = subpool;
          APR_ARRAY_PUSH (db_stack, void *) = item;
        }
      else
        {
          svn_pool_destroy (subpool);
        }

      /*** Step E - Save our state for the next iteration.  If our
           caller opened or added PATH as a directory, that becomes
           our LAST_PATH.  Otherwise, we use PATH's parent
           directory. ***/

      /* NOTE:  The variable LAST_PATH needs to outlive the loop. */
      if (db)
        last_path = path; /* lives in a pool outside our control. */
      else
        last_path = apr_pstrdup (pool, pdir); /* duping into POOL. */
    }

  /* Destroy the iteration subpool. */
  svn_pool_destroy (iterpool);

  /* Close down any remaining open directory batons. */
  while (db_stack->nelts)
    {
      SVN_ERR (pop_stack (db_stack, editor));
    }

  return SVN_NO_ERROR;
}
