#include <sys/stat.h>

#include <limits.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>

#include "request.h"
#include "keyvalue.h"
#include "log.h"

inline int light_isdigit(int c) {
	return (c >= '0' && c <= '9');
}

inline int light_isalpha(int c) {
	return (c >= 'a' && c <= 'z') || 
		(c >= 'A' && c <= 'Z');
}

inline int light_isalnum(int c) {
	return light_isdigit(c) || light_isalpha(c);
}

static int request_check_hostname(server *srv, connection *con, buffer *host) {
	enum { DOMAINLABEL, TOPLABEL } stage = TOPLABEL;
	size_t i;
	int label_len = 0;
	size_t host_len;
	char *colon;
	int is_ip = -1; /* -1 don't know yet, 0 no, 1 yes */ 
	int level = 0;
	
	UNUSED(srv);
	UNUSED(con);

	/*
	 *       hostport      = host [ ":" port ]
	 *       host          = hostname | IPv4address
	 *       hostname      = *( domainlabel "." ) toplabel [ "." ]
	 *       domainlabel   = alphanum | alphanum *( alphanum | "-" ) alphanum
	 *       toplabel      = alpha | alpha *( alphanum | "-" ) alphanum
	 *       IPv4address   = 1*digit "." 1*digit "." 1*digit "." 1*digit
	 *       port          = *digit
	 */
	
	/* no Host: */
	if (!host || host->used == 0) return 0;
	
	host_len = host->used - 1;
	
	if (NULL != (colon = memchr(host->ptr, ':', host_len))) {
		register char *c = colon + 1;
		
		/* check portnumber */
		for (; *c; c++) {
			if (!light_isdigit(*c)) return -1;
		}
		
		/* remove the port from the host-len */
		host_len = colon - host->ptr;
	}
	
	/* Host is empty */
	if (host_len == 0) return -1;
	
	/* scan from the right and skip the \0 */
	for (i = host_len - 1; i + 1 > 0; i--) {
		char c = host->ptr[i];

		switch (stage) {
		case TOPLABEL: 
			if (c == '.') {
				/* only switch stage, if this is not the last character */
				if (i != host_len - 1) {
					if (label_len == 0) {
						return -1;
					}
					
					/* check the first character at right of the dot */
					if (is_ip == 0) {
						if (!light_isalpha(host->ptr[i+1])) {
							return -1; 
						}
					} else if (!light_isdigit(host->ptr[i+1])) {
						is_ip = 0;
					} else if ('-' == host->ptr[i+1]) {
						return -1;
					} else {
						/* just digits */
						is_ip = 1;
					}
						
					stage = DOMAINLABEL;
					
					label_len = 0;
					level++;
				} else if (i == 0) {
					/* just a dot and nothing else is evil */
					return -1;
				}
			} else if (i == 0) {
				/* the first character of the hostname */
				if (!light_isalpha(c)) {
					return -1;
				}
				label_len++;
			} else {
				if (c != '-' && !light_isalnum(c)) {
					return -1;
				}
				if (is_ip == -1) {
					if (!light_isdigit(c)) is_ip = 0;
				}
				label_len++;
			}
			
			break;
		case DOMAINLABEL:
			if (is_ip == 1) {
				if (c == '.') {
					if (label_len == 0) {
						return -1;
					}
					
					label_len = 0;
					level++;
				} else if (!light_isdigit(c)) {
					return -1;
				} else {
					label_len++;
				}
			} else {
				if (c == '.') {
					if (label_len == 0) {
						return -1;
					}
					
					/* c is either - or alphanum here */
					if ('-' == host->ptr[i+1]) {
						return -1;
					}
					
					label_len = 0;
					level++;
				} else if (i == 0) {
					if (!light_isalnum(c)) {
						return -1;
					}
					label_len++;
				} else {
					if (c != '-' && !light_isalnum(c)) {
						return -1;
					}
					label_len++;
				}
			}
			
			break;
		}
	}
	
	/* a IP has to consist of 4 parts */
	if (is_ip == 1 && level != 3) {
		return -1;
	}
	
	if (label_len == 0) {
		return -1;
	}
	
	return 0;
}

#if 0
#define DUMP_HEADER
#endif

int http_request_parse(server *srv, connection *con) {
	char *uri = NULL, *proto = NULL, *method = NULL, con_length_set, keep_alive_set = 0;
	int is_key = 1, key_len = 0, is_ws_after_key = 0, in_folding;
	char *value = NULL, *key = NULL;
	
	int line = 0;
	
	int request_line_stage = 0;
	size_t i, first;
	
	int done = 0;
	
	data_string *ds = NULL;
	
	/* 
	 * Request: "^(GET|POST|HEAD) ([^ ]+(\\?[^ ]+|)) (HTTP/1\\.[01])$" 
	 * Option : "^([-a-zA-Z]+): (.+)$"                    
	 * End    : "^$"
	 */
#ifdef DUMP_HEADER
	log_error_write(srv, __FILE__, __LINE__, "dbd", 
			con->fd, con->request.request, con->request.request->used);
#endif
	/* fill the local request buffer */
	buffer_copy_string_buffer(con->parse_request, con->request.request);
	
	/* \r\n -> \n\0 */
	
	keep_alive_set = 0;
	con_length_set = 0;

	for (i = 0, first = 0; i < con->parse_request->used && line == 0; i++) {
		char *cur = con->parse_request->ptr + i;
		
		switch(*cur) {
		case '\r': 
			if (con->parse_request->ptr[i+1] == '\n') {
				http_method_t r;
				char *nuri = NULL;
				
				con->parse_request->ptr[i] = '\0';
				con->parse_request->ptr[i+1] = '\0';
				
				buffer_copy_string_len(con->request.request_line, con->parse_request->ptr, i);
				
				if (request_line_stage != 2) {
					con->http_status = 400;
					con->response.keep_alive = 0;
					con->keep_alive = 0;
					
					return 0;
				}
				
				proto = con->parse_request->ptr + first;
				
				*(uri - 1) = '\0';
				*(proto - 1) = '\0';
				
				/* we got the first one :) */
				if (-1 == (r = get_http_method_key(method))) {
					con->http_status = 501;
					con->response.keep_alive = 0;
					con->keep_alive = 0;
					
					
					log_error_write(srv, __FILE__, __LINE__, "sb", 
							"unknown http-method", con->request.request);
					
					return 0;
				}
				
				con->request.http_method = r;
				
				if (0 == strncmp(proto, "HTTP/1.", 7)) {
					if (proto[7] == '1') {
						con->request.http_version = HTTP_VERSION_1_1;
					} else if (proto[7] == '0') {
						con->request.http_version = HTTP_VERSION_1_0;
					} else { 
						con->http_status = 505;
						return 0;
					}
				} else {
					con->http_status = 400;
					con->keep_alive = 0;
					return 0;
				}
				
				if (0 == strncmp(uri, "http://", 7) &&
				    NULL != (nuri = strchr(uri + 7, '/'))) {
					/* ignore the host-part */
					
					buffer_copy_string_len(con->request.uri, nuri, proto - nuri - 1);
				} else {
					/* everything looks good so far */
					buffer_copy_string_len(con->request.uri, uri, proto - uri - 1);
				}
				
				con->http_status = 200;
				
				i++;
				line++;
				first = i+1;
			}
			break;
		case ' ':
			switch(request_line_stage) {
			case 0: 
				method = con->parse_request->ptr + first; 
				first = i + 1;
				break;
			case 1:
				uri = con->parse_request->ptr + first; 
				first = i + 1;
				break;
			default:
				/* ERROR */
				con->http_status = 400;
				con->response.keep_alive = 0;
				con->keep_alive = 0;
				
				return 0;
			}
			
			request_line_stage++;
			break;
		}
	}
	
	in_folding = 0;
	
	for (; i < con->parse_request->used && !done; i++) {
		char *cur = con->parse_request->ptr + i;
		
		if (is_key) {
			size_t j;
			int got_colon = 0;
			
			/**
			 * 1*<any CHAR except CTLs or separators>
			 * CTLs == 0-31 + 127
			 * 
			 */
			switch(*cur) {
			case ':':
				is_key = 0;
				
				value = cur + 1;
				
				if (is_ws_after_key == 0) {
					key_len = i - first;
				}
				is_ws_after_key = 0;
					
				break;
			case '(':
			case ')':
			case '<':
			case '>':
			case '@':
			case ',':
			case ';':
			case '\\':
			case '\"':
			case '/':
			case '[':
			case ']':
			case '?':
			case '=':
			case '{':
			case '}':
				con->http_status = 400;
				con->keep_alive = 0;
				con->response.keep_alive = 0;
				
				log_error_write(srv, __FILE__, __LINE__, "sbsd", 
						"invalid character in key", con->request.request, cur, *cur);
				return 0;
			case ' ':
			case '\t':
				if (i == first) {
					log_error_write(srv, __FILE__, __LINE__, "s", "(debug) Line Folding");
					
					is_key = 0;
					in_folding = 1;
					value = cur;
					
					break;
				}
				
				
				key_len = i - first;
				
				/* skip every thing up to the : */
				for (j = 1; !got_colon; j++) {
					switch(con->parse_request->ptr[j + i]) {
					case ' ':
					case '\t':
						/* skip WS */
						continue;
					case ':':
						/* ok, done */
						
						i += j - 1;
						got_colon = 1;
						
						break;
					default:
						/* error */
						
						log_error_write(srv, __FILE__, __LINE__, "sb", 
							"WS character in key", con->request.request);
					
						con->http_status = 400;
						con->response.keep_alive = 0;
						con->keep_alive = 0;
						
						return 0;
					}
				}
				
				break;
			case '\r':
				if (con->parse_request->ptr[i+1] == '\n' && i == first) {
					/* End of Header */
					con->parse_request->ptr[i] = '\0';
					con->parse_request->ptr[i+1] = '\0';
					
					i++;
					
					done = 1;
					
					break;
				} else {
					log_error_write(srv, __FILE__, __LINE__, "sb", 
							"CR without LF", con->request.request);
					
					con->http_status = 400;
					con->keep_alive = 0;
					con->response.keep_alive = 0;
					return 0;
				}
				/* fall thru */
			case 0: /* illegal characters (faster than a if () :) */
			case 1:
			case 2:
			case 3:
			case 4:
			case 5:
			case 6:
			case 7:
			case 8:
			case 10:
			case 11:
			case 12:
			case 14:
			case 15:
			case 16:
			case 17:
			case 18:
			case 19:
			case 20:
			case 21:
			case 22:
			case 23:
			case 24:
			case 25:
			case 26:
			case 27:
			case 28:
			case 29:
			case 30:
			case 31:
			case 127:
				con->http_status = 400;
				con->keep_alive = 0;
				con->response.keep_alive = 0;
				
				log_error_write(srv, __FILE__, __LINE__, "sbsd", 
						"CTL character in key", con->request.request, cur, *cur);
				
				return 0;
			default:
				/* ok */
				break;
			}
		} else {
			switch(*cur) {
			case '\r': 
				if (con->parse_request->ptr[i+1] == '\n') {
					/* End of Headerline */
					con->parse_request->ptr[i] = '\0';
					con->parse_request->ptr[i+1] = '\0';
					
					if (in_folding) {
						if (!ds) {
							/* 400 */
					
							log_error_write(srv, __FILE__, __LINE__, "sb", 
									"WS at the start of first line", con->request.request);
					
							con->http_status = 400;
							con->keep_alive = 0;
							con->response.keep_alive = 0;
							return 0;
						}
						buffer_append_string(ds->value, value);
					} else {
						int s_len;
						key = con->parse_request->ptr + first;
					
						s_len = cur - value;
						
						if (s_len > 0) {
							int cmp = 0;
							if (NULL == (ds = (data_string *)array_get_unused_element(con->request.headers, TYPE_STRING))) {
								ds = data_string_init();
							}
							buffer_copy_string_len(ds->key, key, key_len);
							buffer_copy_string_len(ds->value, value, s_len);
							
							/* retreive values 
							 * 
							 * 
							 * the list of options is sorted to simplify the search
							 */
							
							if (0 == (cmp = buffer_caseless_compare(CONST_BUF_LEN(ds->key), CONST_STR_LEN("Connection")))) {
								/* Fixme: this should be case-insensitive  */
								con->keep_alive = (NULL != strstr(ds->value->ptr, "Keep-Alive")) ? 1 : 0;
								keep_alive_set = 1;
							} else if (cmp > 0 && 0 == (cmp = buffer_caseless_compare(CONST_BUF_LEN(ds->key), CONST_STR_LEN("Content-Length")))) {
								char *err;
								unsigned long int r;
								size_t j;
								
								if (ds->value->used == 0) SEGFAULT();
								
								for (j = 0; j < ds->value->used - 1; j++) {
									char c = ds->value->ptr[j];
									if (!isdigit((unsigned char)c)) {
										log_error_write(srv, __FILE__, __LINE__, "sb", "content-length broken:", ds->value);
										
										con->http_status = 400;
										con->keep_alive = 0;
										
										array_insert_unique(con->request.headers, (data_unset *)ds);
										return 0;
									}
								}
								
								r = strtoul(ds->value->ptr, &err, 10);
								
								if (*err == '\0') {
									con_length_set = 1;
									con->request.content_length = r;
								} else {
									log_error_write(srv, __FILE__, __LINE__, "sb", "content-length broken:", ds->value);
									
									con->http_status = 400;
									con->keep_alive = 0;
									
									array_insert_unique(con->request.headers, (data_unset *)ds);
									return 0;
								}
							} else if (cmp > 0 && 0 == (cmp = buffer_caseless_compare(CONST_BUF_LEN(ds->key), CONST_STR_LEN("Content-Type")))) {
								con->request.http_content_type = ds->value->ptr;
							} else if (cmp > 0 && 0 == (cmp = buffer_caseless_compare(CONST_BUF_LEN(ds->key), CONST_STR_LEN("Expect")))) {
								/* HTTP 2616 8.2.3 
								 * Expect: 100-continue
								 * 
								 *   -> (10.1.1)  100 (read content, process request, send final status-code)
								 *   -> (10.4.18) 417 (close)
								 * 
								 * (not handled at all yet, we always send 417 here)
								 */
								
								con->http_status = 417;
								con->keep_alive = 0;
								
								array_insert_unique(con->request.headers, (data_unset *)ds);
								return 0;
							} else if (cmp > 0 && 0 == (cmp = buffer_caseless_compare(CONST_BUF_LEN(ds->key), CONST_STR_LEN("Host")))) {
								con->request.http_host = ds->value;
							} else if (cmp > 0 && 0 == (cmp = buffer_caseless_compare(CONST_BUF_LEN(ds->key), CONST_STR_LEN("If-Modified-Since")))) {
								con->request.http_if_modified_since = ds->value->ptr;
							} else if (cmp > 0 && 0 == (cmp = buffer_caseless_compare(CONST_BUF_LEN(ds->key), CONST_STR_LEN("If-None-Match")))) {
								con->request.http_if_none_match = ds->value->ptr;
							} else if (cmp > 0 && 0 == (cmp = buffer_caseless_compare(CONST_BUF_LEN(ds->key), CONST_STR_LEN("Range")))) {
								/* bytes=.*-.* */
								
								if (0 == strncasecmp(ds->value->ptr, "bytes=", 6) &&
								    NULL != strchr(ds->value->ptr+6, '-')) {
									
									con->request.http_range = ds->value->ptr + 6;
								}
							}
							
							array_insert_unique(con->request.headers, (data_unset *)ds);
						} else {
							/* empty header-fields are not allowed by HTTP-RFC, we just ignore them */
						}
					}
					
					i++;
					first = i+1;
					is_key = 1;
					value = 0;
					key_len = 0;
					in_folding = 0;
				} else {
					log_error_write(srv, __FILE__, __LINE__, "sb", 
							"CR without LF", con->request.request);
					
					con->http_status = 400;
					con->keep_alive = 0;
					con->response.keep_alive = 0;
					return 0;
				}
				break;
			case ' ':
			case '\t':
				/* strip leading WS */
				if (value == cur) value = cur+1;
			default:
				break;
			}
		}
	}
	
	con->header_len = i;
	
	/* do some post-processing */

	if (con->request.http_version == HTTP_VERSION_1_1) {
		if (keep_alive_set == 0) {
			/* no Connection-Header sent */
			
			/* HTTP/1.1 -> keep-alive default TRUE */
			con->keep_alive = 1;
		}
		
		/* RFC 2616, 14.23 */
		if (con->request.http_host == NULL) {
			con->http_status = 400;
			con->response.keep_alive = 0;
			con->keep_alive = 0;
			return 0;
		}
	} else {
		if (keep_alive_set == 0) {
			/* no Connection-Header sent */
			
			/* HTTP/1.0 -> keep-alive default FALSE  */
			con->keep_alive = 0;
		}
	}
	
	/* check hostname field */
	if (0 != request_check_hostname(srv, con, con->request.http_host)) {
		log_error_write(srv, __FILE__, __LINE__, "sb", 
				"(notice) Bad Hostname:", con->request.http_host);
		
		con->http_status = 400;
		con->response.keep_alive = 0;
		con->keep_alive = 0;
		
		return 0;
	}
	
	/* check if we have read post data */
	if (con->request.http_method == HTTP_METHOD_POST) {
		server_socket *srv_socket = con->srv_socket;
		if (con->request.http_content_type == NULL) {
			log_error_write(srv, __FILE__, __LINE__, "dsb", 
					con->fd, "(notice) POST request, but content-type not set",
					con->request.request);
		}
		
		if (con_length_set == 0) {
			/* content-length is missing */
			
			con->http_status = 411;
			return 0;
		}
		
		/* don't handle more the SSIZE_MAX bytes in content-length */
		if (con->request.content_length > SSIZE_MAX) {
			con->http_status = 413;
			return 0;
		}
		
		/* divide by 1024 as srvconf.max_request_size is in kBytes */
		if (srv_socket->max_request_size != 0 &&
		    (con->request.content_length >> 10) > srv_socket->max_request_size) {
			/* the request body itself is larger then 
			 * our our max_request_size
			 */
		
			con->http_status = 413;
		
			return 0;
		}
		
		
		/* we have content */
		if (con->request.content_length != 0) {
			return 1;
		}
	}
	
	return 0;
}

int http_request_header_finished(server *srv, connection *con) {
	UNUSED(srv);

	if (con->request.request->used < 5) return 0;
	
	if (0 == memcmp(con->request.request->ptr + con->request.request->used - 5, "\r\n\r\n", 4)) return 1;
	if (NULL != strstr(con->request.request->ptr, "\r\n\r\n")) return 1;
	
	return 0;
}
