#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <stdio.h>

#include "base.h"
#include "log.h"
#include "buffer.h"
#include "response.h"

#include "plugin.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef HAVE_LOCALIZER_H	
#include <localizer.h>
#endif


/* plugin config for all request/connections */

typedef struct {
	PLUGIN_DATA;
	buffer *params;
	buffer *conf_db_path;
	buffer *conf_base_url;
	
	array *conf_allowed_ips;
#ifdef HAVE_LOCALIZER_H
	localizer *l;
#endif
} plugin_data;

/* init the plugin data */
INIT_FUNC(mod_localizer_init) {
	plugin_data *p;
	
	p = calloc(1, sizeof(*p));
	
	p->params = buffer_init();
	p->conf_db_path = buffer_init();
	p->conf_base_url = buffer_init();
	
	p->conf_allowed_ips = array_init();
#ifdef HAVE_LOCALIZER_H
	p->l = localizer_init();
#endif
	
	return p;
}

/* detroy the plugin data */
FREE_FUNC(mod_localizer_free) {
	plugin_data *p = p_d;
	UNUSED(srv);
	
	if (!p) return HANDLER_GO_ON;
	
	buffer_free(p->params);
	buffer_free(p->conf_db_path);
	buffer_free(p->conf_base_url);
	
	array_free(p->conf_allowed_ips);
#ifdef HAVE_LOCALIZER_H	
	localizer_free(p->l);
#endif
	
	free(p);
	
	return HANDLER_GO_ON;
}

/* handle plugin config and check values */

SETDEFAULTS_FUNC(mod_localizer_set_defaults) {
	plugin_data *p = p_d;
	size_t i = 0;
	
	config_values_t cv[] = { 
		{ "localizer.database-path",    NULL, T_CONFIG_STRING, T_CONFIG_SCOPE_CONNECTION },       /* 0 */
		{ "localizer.base-url",         NULL, T_CONFIG_STRING, T_CONFIG_SCOPE_CONNECTION },       /* 1 */
		{ "localizer.allowed-ips",      NULL, T_CONFIG_ARRAY, T_CONFIG_SCOPE_CONNECTION },        /* 2 */
		{ NULL,                         NULL, T_CONFIG_UNSET, T_CONFIG_SCOPE_UNSET }
	};
	
	if (!p) return HANDLER_ERROR;
	
	/* 0 */
	cv[i++].destination = p->conf_db_path;
	cv[i++].destination = p->conf_base_url;
	cv[i++].destination = p->conf_allowed_ips;
	
	if (0 != config_insert_values(srv, cv)) {
		return HANDLER_ERROR;
	}
	
	if (0 == p->conf_db_path->used) {
		log_error_write(srv, __FILE__, __LINE__, "s", 
				"localizer.database-path is empty and has to be set");
		
		return HANDLER_ERROR;
	}
	
	if (0 == p->conf_base_url->used) {
		log_error_write(srv, __FILE__, __LINE__, "s", 
				"localizer.base-url is empty and has to be set");
		
		return HANDLER_ERROR;
	}
	
	if (0 == p->conf_allowed_ips->used) {
		log_error_write(srv, __FILE__, __LINE__, "s", 
				"localizer.allowed-ips is empty and has to be set");
		
		return HANDLER_ERROR;
	}
#ifdef HAVE_LOCALIZER_H	
	if (-1 == localizer_read(p->l, p->conf_db_path->ptr)) {
		log_error_write(srv, __FILE__, __LINE__, "sbs", 
				"localizer_read:", p->conf_db_path, strerror(errno));
		
		return HANDLER_ERROR;
	}
	
	return HANDLER_GO_ON;
#else
	log_error_write(srv, __FILE__, __LINE__, "ss", 
			"mod_localizer:", "liblocalizer and/or headers are not installed");
	
	return HANDLER_ERROR;
#endif
}

#ifdef HAVE_LOCALIZER_H	
URIHANDLER_FUNC(mod_localizer_uri_handler) {
	plugin_data *p = p_d;
	char *format = NULL, *ip = NULL, *eq = NULL, *param, *amp;
	long l_ip;
	
	l_data_export lext;
	char buf[32];
	
	buffer *b;
	
	if (con->uri.path->used == 0) return HANDLER_GO_ON;
	
	if (!buffer_is_equal(con->uri.path, p->conf_base_url)) return HANDLER_GO_ON;
	
	/* check if remote ip is allowed to access this URL */
	if (con->dst_addr.plain.sa_family == AF_INET) {
		char *c = inet_ntoa(con->dst_addr.ipv4.sin_addr);
		size_t i;
		
		/* iterate array */
		
		for (i = 0; i < p->conf_allowed_ips->used; i++) {
			data_string *ds = ((data_string **)(p->conf_allowed_ips->data))[i];
			
			if (ds->value->used == 0) continue;
			
			if (0 == strcmp(ds->value->ptr, c)) {
				break;
			}
		}
		
		if (i == p->conf_allowed_ips->used) {
			log_error_write(srv, __FILE__, __LINE__, "sss", 
					"mod_localizer:", "host is not allowed to connect:", c);
			
			con->http_status = 403;
			
			return HANDLER_FINISHED;
			
		} 
	} else {
		/* IPv6 is not supported */
		con->http_status = 500;
		
		log_error_write(srv, __FILE__, __LINE__, "ss", 
				"mod_localizer:", "IPv6 is not supported");
		
		return HANDLER_FINISHED;
	}
	
	
	buffer_copy_string_buffer(p->params, con->uri.query);
	
	/* search for & */
	for (param = p->params->ptr; NULL != (amp = strchr(param, '&')); param = amp + 1) {
		*amp = '\0';
		
		/* search for = in param */
		
		if (NULL == (eq = strchr(param, '='))) {
			con->http_status = 500;
			
			log_error_write(srv, __FILE__, __LINE__, "ssb", 
				"mod_localizer:", "= is missing", con->uri.path);
			
			return HANDLER_FINISHED;
		}
		
		*eq++ = '\0';
		
		if (0 == strcmp(param, "ip")) {
			ip = eq;
		} else if (0 == strcmp(param, "format")) {
			format = eq;
		}
	}
	
	if (NULL == (eq = strchr(param, '='))) {
		con->http_status = 500;
		
		log_error_write(srv, __FILE__, __LINE__, "ssb", 
				"mod_localizer:", "= is missing", con->uri.path);
		
		return HANDLER_FINISHED;
	}
		
	*eq++ = '\0';
		
	if (0 == strcmp(param, "ip")) {
		ip = eq;
	} else if (0 == strcmp(param, "format")) {
		format = eq;
	}
	
	if (!ip) {
		log_error_write(srv, __FILE__, __LINE__, "ssb", 
				"mod_localizer:", "'ip' field in URL is missing", con->uri.path);
		
		con->http_status = 500;
		
		return HANDLER_FINISHED;
	}
	
	
	if (0 == (l_ip = localizer_ip2int(ip))) {
		con->http_status = 500;
		
		log_error_write(srv, __FILE__, __LINE__, "ssb", 
				"mod_localizer:", "localizer_ip2int failed on", con->uri.path);
		
		return HANDLER_FINISHED;
	}
	
	if (-1 == (localizer_search(p->l, l_ip, &lext))) {
		con->http_status = 500;
		
		log_error_write(srv, __FILE__, __LINE__, "ssb", 
				"mod_localizer:", "localizer_search failed on", con->uri.path);
		
		return HANDLER_FINISHED;
	}
	
	b = chunkqueue_get_append_buffer(con->write_queue);
	buffer_reset(b);
	
	if ((format == NULL) || (0 == strcmp(format, "html"))) {
#define TABLE_START() \
	buffer_append_string(b, \
		"<?xml version=\"1.0\" encoding=\"iso-8859-1\"?>\n" \
		      "<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\"\n" \
		      "    \"http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd\">\n" \
		      "<html xmlns=\"http://www.w3.org/1999/xhtml\" xml:lang=\"en\" lang=\"en\">\n" \
		      " <head>\n" \
		      "  <title>localizer</title>\n" \
		      " </head>\n" \
		      " <body>\n" \
		      "  <table>\n");
#define TABLE_END() \
	buffer_append_string(b, "  </table>\n </body>\n</html>");
#define TABLE_ROW(x, y) \
	buffer_append_string(b, "   <tr><td>" x "</td><td>"); \
	buffer_append_string(b, lext.y); \
	buffer_append_string(b, "</td></tr>\n") ;
		
#define TABLE_ROW_DOUBLE(x, y) \
	buffer_append_string(b, "   <tr><td>" x "</td><td>"); \
	sprintf(buf, "%.4f", lext.y); \
	buffer_append_string(b, buf); \
	buffer_append_string(b, "</td></tr>\n");
		
		response_header_overwrite(srv,con, CONST_STR_LEN("Content-Type"), CONST_STR_LEN("text/html"));
		
		TABLE_START();
		
		TABLE_ROW("City", city);
		TABLE_ROW("Province", province);
		TABLE_ROW("Country", country);
		TABLE_ROW("Provider", provider);
		
		TABLE_ROW_DOUBLE("Longitude", longitude);
		TABLE_ROW_DOUBLE("Latitude", latitude);
		TABLE_ROW_DOUBLE("Radius", distance);
		
		TABLE_END();
		
#undef TABLE_START
#undef TABLE_END
#undef TABLE_ROW
#undef TABLE_ROW_DOUBLE
		
	} else if (0 == strcmp(format, "csv")) {
		int line = 0;
#define TABLE_START() \
	buffer_append_string(b, "## City, Province, Country, Provider, Longitude, Latitude, Radius\n");
#define TABLE_END() \
	buffer_append_string(b, "\n");
#define TABLE_ROW(x, y) \
	if (line++) buffer_append_string(b, ",");  \
	buffer_append_string(b, lext.y); 
	
		
#define TABLE_ROW_DOUBLE(x, y) \
	if (line++) buffer_append_string(b, ",");  \
	sprintf(buf, "%.4f", lext.y); \
	buffer_append_string(b, buf); 
	
		response_header_overwrite(srv,con, CONST_STR_LEN("Content-Type"), CONST_STR_LEN("text/csv"));
		
		TABLE_START();
		
		TABLE_ROW("City", city);
		TABLE_ROW("Province", province);
		TABLE_ROW("Country", country);
		TABLE_ROW("Provider", provider);
		
		TABLE_ROW_DOUBLE("Longitude", longitude);
		TABLE_ROW_DOUBLE("Latitude", latitude);
		TABLE_ROW_DOUBLE("Radius", distance);
		
		TABLE_END();
		
#undef TABLE_START
#undef TABLE_END
#undef TABLE_ROW
#undef TABLE_ROW_DOUBLE
	} else if (0 == strcmp(format, "text")) {
#define TABLE_START() ;
		
#define TABLE_END() ;
		
#define TABLE_ROW(x, y) \
	buffer_append_string(b, x" = " ); \
	buffer_append_string(b, lext.y); \
	buffer_append_string(b, "\n");
#define TABLE_ROW_DOUBLE(x, y) \
	buffer_append_string(b, x" = " ); \
	sprintf(buf, "%.4f", lext.y); \
	buffer_append_string(b, buf); \
	buffer_append_string(b, "\n");
	
		response_header_overwrite(srv,con, CONST_STR_LEN("Content-Type"), CONST_STR_LEN("text/plain"));
		
		TABLE_START();
		
		TABLE_ROW("City", city);
		TABLE_ROW("Province", province);
		TABLE_ROW("Country", country);
		TABLE_ROW("Provider", provider);
		
		TABLE_ROW_DOUBLE("Longitude", longitude);
		TABLE_ROW_DOUBLE("Latitude", latitude);
		TABLE_ROW_DOUBLE("Radius", distance);
		
		TABLE_END();
		
#undef TABLE_START
#undef TABLE_END
#undef TABLE_ROW
#undef TABLE_ROW_DOUBLE
	}
	
	con->file_finished = 1;
	
	/* not found */
	return HANDLER_FINISHED;
}
#endif
/* this function is called at dlopen() time and inits the callbacks */

int mod_localizer_plugin_init(plugin *p) {
	p->name        = buffer_init_string("localizer");
	
	p->init        = mod_localizer_init;
#ifdef HAVE_LOCALIZER_H	
	p->handle_uri_clean  = mod_localizer_uri_handler;
#endif
	p->set_defaults  = mod_localizer_set_defaults;
	p->cleanup     = mod_localizer_free;
	
	p->data        = NULL;
	
	return 0;
}
