/*
 * $Id: ioperm.c,v 1.2 2003/01/21 20:42:54 telka Exp $
 *
 * ioperm driver implementation
 * Copyright (C) 2002 ETC s.r.o.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 *
 * Written by Marcel Telka <marcel@telka.sk>, 2002.
 *
 * Documentation:
 * [1] Dale Roberts, "Direct Port I/O and Windows NT", Dr. Dobb's Journal,
 *     May 1996
 *
 * Note: This allow access to I/O ports for non-root processes too.
 *
 */

#include <ntddk.h>

#define	IOCTL_IOPERM	CTL_CODE( FILE_DEVICE_UNKNOWN, 0xA00, METHOD_BUFFERED, FILE_ANY_ACCESS )

struct ioperm_data {
	unsigned long from;
	unsigned long num;
	int turn_on;
};

/* undocumented functions - see [1] */
void STDCALL Ke386IoSetAccessProcess( PEPROCESS, int );
void STDCALL Ke386QueryIoAccessMap( int, unsigned char * );
void STDCALL Ke386SetIoAccessMap( int, unsigned char * );

static VOID STDCALL
ioperm_unload( IN PDRIVER_OBJECT DriverObject )
{
	UNICODE_STRING SymbolicLinkName;	

	RtlInitUnicodeString( &SymbolicLinkName, L"\\DosDevices\\ioperm" );
	IoDeleteSymbolicLink( &SymbolicLinkName );
	IoDeleteDevice( DriverObject->DeviceObject );
}

static NTSTATUS STDCALL
ioperm_dispatch_create( IN PDEVICE_OBJECT DeviceObject, IN PIRP Irp )
{
	Irp->IoStatus.Status = STATUS_SUCCESS;
	Irp->IoStatus.Information = 0;
	IoCompleteRequest( Irp, IO_NO_INCREMENT );

	return STATUS_SUCCESS;
}

/* workaround for bug 666316 and bug 671964 */
#undef IoGetCurrentIrpStackLocation
#define	IoGetCurrentIrpStackLocation(irp) (*(struct _IO_STACK_LOCATION**)((char*)(irp) + 96))

static NTSTATUS STDCALL
ioperm_dispatch_device_control( IN PDEVICE_OBJECT DeviceObject, IN PIRP Irp )
{
	PIO_STACK_LOCATION io_stack;

	/* default results */
	Irp->IoStatus.Status = STATUS_NOT_IMPLEMENTED;
	Irp->IoStatus.Information = 0;

	/* get current I/O stack location */
	io_stack = IoGetCurrentIrpStackLocation( Irp );

	/* test valid IOCTL code */
	if (io_stack
		&& (io_stack->MajorFunction == IRP_MJ_DEVICE_CONTROL)
		&& (io_stack->Parameters.DeviceIoControl.IoControlCode == IOCTL_IOPERM)
		&& (io_stack->Parameters.DeviceIoControl.InputBufferLength >= sizeof (struct ioperm_data))) {
		/* get input buffer */
		struct ioperm_data *ioperm_data = (struct ioperm_data *) Irp->AssociatedIrp.SystemBuffer;

		/* test input buffer size and parameters */
		if ((io_stack->Parameters.DeviceIoControl.InputBufferLength < sizeof (struct ioperm_data))
			|| (!ioperm_data) || (ioperm_data->from + ioperm_data->num > 0x400))
		{
			Irp->IoStatus.Status = STATUS_INVALID_PARAMETER;
		} else {
			/* allocate buffer for I/O permission map */
			unsigned char *ioperms = MmAllocateNonCachedMemory( 0x2000 );
			if (!ioperms) {
				Irp->IoStatus.Status = STATUS_INSUFFICIENT_RESOURCES;
			} else {
				unsigned long i;

				/* enable I/O access for process */
				Ke386IoSetAccessProcess( PsGetCurrentProcess(), 1 );
				/* query actual I/O permission map */
				Ke386QueryIoAccessMap( 1, ioperms );

				/* change I/O permission map */
				for (i = ioperm_data->from; i < ioperm_data->from + ioperm_data->num; i++) {
					if (ioperm_data->turn_on) {
						ioperms[i / 8] &= ~(1 << (i % 8));
					} else {
						ioperms[i / 8] |= (1 << (i % 8));
					}
				}

				/* setup new I/O permission map */
				Ke386SetIoAccessMap( 1, ioperms );

				/* free buffer for I/O permission map */
				MmFreeNonCachedMemory( ioperms, 0x2000 );

				Irp->IoStatus.Status = STATUS_SUCCESS;
				Irp->IoStatus.Information = sizeof (struct ioperm_data);
			}
		}
	}

	/* complete IRP request */
	IoCompleteRequest( Irp, IO_NO_INCREMENT );

	return Irp->IoStatus.Status;
}

NTSTATUS STDCALL
DriverEntry( IN PDRIVER_OBJECT DriverObject, IN PUNICODE_STRING RegistryPath )
{
	NTSTATUS status;
	PDEVICE_OBJECT DeviceObject;
	UNICODE_STRING DeviceName;
	UNICODE_STRING SymbolicLinkName;

	/* support for service stopping */
	DriverObject->DriverUnload = ioperm_unload;
	/* create support */
	DriverObject->MajorFunction[IRP_MJ_CREATE] = ioperm_dispatch_create;
	/* IOCTL support */
	DriverObject->MajorFunction[IRP_MJ_DEVICE_CONTROL] = ioperm_dispatch_device_control;

	/* initialize counted unicode strings */
	RtlInitUnicodeString( &DeviceName, L"\\Device\\ioperm" );
	RtlInitUnicodeString( &SymbolicLinkName, L"\\DosDevices\\ioperm" );

	/* create device object */
	status = IoCreateDevice( DriverObject, 0, &DeviceName, FILE_DEVICE_UNKNOWN, 0, FALSE, &DeviceObject );
	if (!NT_SUCCESS( status ))
		return status;

	/* create user-visible name for the device */
	status = IoCreateSymbolicLink( &SymbolicLinkName, &DeviceName );
	if (!NT_SUCCESS( status ))
		return status;

	return STATUS_SUCCESS;
}
