/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
 * This is GNU GO, a Go program. Contact gnugo@gnu.org, or see       *
 * http://www.gnu.org/software/gnugo/ for more information.          *
 *                                                                   *
 * Copyright 1999, 2000, 2001, 2002 by the Free Software Foundation. *
 *                                                                   *
 * This program is free software; you can redistribute it and/or     *
 * modify it under the terms of the GNU General Public License as    *
 * published by the Free Software Foundation - version 2             *
 *                                                                   *
 * This program is distributed in the hope that it will be useful,   *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of    *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     *
 * GNU General Public License in file COPYING for more details.      *
 *                                                                   *
 * You should have received a copy of the GNU General Public         *
 * License along with this program; if not, write to the Free        *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,           *
 * Boston, MA 02111, USA.                                            *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/* The functions in this file implements a go board with incremental
 * update of strings and liberties.
 * 
 * See the Texinfo documentation (Utility Functions: Incremental Board)
 * for an introduction.
 */

#include "gnugo.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>

#include "liberty.h"
#include "hash.h"
#include "sgftree.h"
#include "gg_utils.h"

#define KOMASTER_SCHEME 5
#define KOMASTER_TRACE 0

int tracing_inside_komaster = 0;


/* This can be used for internal checks w/in board.c that should
 * typically not be necessary (for speed). */
#if 1
#define PARANOID1(x, pos) ASSERT1(x, pos)
#else
#define PARANOID1(x, pos)
#endif

/* ================================================================ */
/*                          data structures                         */
/* ================================================================ */


/* Incremental string data. */
struct string_data {
  int color;                       /* Color of string, BLACK or WHITE */
  int size;                        /* Number of stones in string. */
  int origin;                      /* Coordinates of "origin", i.e. */
                                   /* "upper left" stone. */
  int liberties;                   /* Number of liberties. */
  int libs[MAX_LIBERTIES];         /* Coordinates of liberties. */
  int neighbors;                   /* Number of neighbor strings */
  int neighborlist[MAXCHAIN];      /* List of neighbor string numbers. */
  int mark;                        /* General purpose mark. */
};


/* we keep the adress and the old value */
struct change_stack_entry {
  int *address;
  int value;
};

/* we keep the adress and the old value */
struct vertex_stack_entry {
  Intersection *address;
  int value;
};


/* Experimental results show that the average number of change stack
 * entries per move usually is in the 20-30 range and very seldom
 * exceeds 40. But since we have no way to recover from running out of
 * stack space, we allocate with a substantial safety margin.
 */
#define STACK_SIZE 80 * MAXSTACK


#define CLEAR_STACKS()\
(change_stack_pointer = change_stack, \
 vertex_stack_pointer = vertex_stack)

/* Begin a record : adress == NULL */
#define BEGIN_CHANGE_RECORD()\
((change_stack_pointer++)->address = NULL,\
 (vertex_stack_pointer++)->address = NULL)

/* Save a value : store the adress and the value in the stack */
#define PUSH_VALUE(v)\
(change_stack_pointer->address = &(v),\
 (change_stack_pointer++)->value = (v))

/* Save a board value : store the adress and the value in the stack */
#define PUSH_VERTEX(v)\
(vertex_stack_pointer->address = &(v),\
 (vertex_stack_pointer++)->value = (v))

#define POP_MOVE()\
  while ((--change_stack_pointer)->address)\
  *(change_stack_pointer->address) =\
  change_stack_pointer->value


#define POP_VERTICES()\
  while ((--vertex_stack_pointer)->address)\
  *(vertex_stack_pointer->address) =\
  vertex_stack_pointer->value


/* ================================================================ */
/*                      static data structures                      */
/* ================================================================ */


/* Main array of string information. */
static struct string_data  string[MAX_STRINGS];

/* Stacks and stack pointers. */
static struct change_stack_entry change_stack[STACK_SIZE];
static struct change_stack_entry *change_stack_pointer;

static struct vertex_stack_entry vertex_stack[STACK_SIZE];
static struct vertex_stack_entry *vertex_stack_pointer;


/* Index into list of strings. The index is only valid if there is a
 * stone at the vertex.
 */
static int string_number[BOARDMAX];


/* The stones in a string are linked together in a cyclic list. 
 * These are the coordinates to the next stone in the string.
 */
static int next_stone[BOARDMAX];


/* ---------------------------------------------------------------- */


/* Macros to traverse the stones of a string.
 *
 * Usage:
 * int s, pos;
 * s = find_the_string()
 * pos = FIRST_STONE(s);
 *   do {
 *    use_stone(pos);
 *    pos = NEXT_STONE(pos);
 *  } while (!BACK_TO_FIRST_STONE(s, pos));
 */
#define FIRST_STONE(s) \
  (string[s].origin)

#define NEXT_STONE(pos) \
  (next_stone[pos])

#define BACK_TO_FIRST_STONE(s, pos) \
  ((pos) == string[s].origin)


/* Assorted useful macros.
 *
 * Some of them could have been functions but are implemented as
 * macros for speed.
 */

#define LIBERTY(pos) \
  (board[pos] == EMPTY)

#define UNMARKED_LIBERTY(pos) \
  (board[pos] == EMPTY && ml[pos] != liberty_mark)

#define MARK_LIBERTY(pos) \
  ml[pos] = liberty_mark

#define UNMARKED_STRING(pos) \
  (string[string_number[pos]].mark != string_mark)

#define UNMARKED_OPPONENT_STRING(s, pos)\
  (board[pos] == OTHER_COLOR(string[s].color)\
   && string[string_number[pos]].mark != string_mark)

#define UNMARKED_OWN_STRING(s, pos)\
  (board[pos] == string[s].color\
   && string[string_number[pos]].mark != string_mark)

#define MARK_STRING(pos) string[string_number[pos]].mark = string_mark

#define STRING_AT_VERTEX(pos, s)\
  (IS_STONE(board[pos]) && string_number[pos] == (s))
  
#define LIBERTIES(pos)\
  string[string_number[pos]].liberties

#define COUNTSTONES(pos) \
  string[string_number[pos]].size

#define ADD_LIBERTY(s, pos)\
  do {\
    if (string[s].liberties < MAX_LIBERTIES)\
      string[s].libs[string[s].liberties] = pos;\
    string[s].liberties++;\
  } while (0)

#define ADD_AND_MARK_LIBERTY(s, pos)\
  do {\
    if (string[s].liberties < MAX_LIBERTIES)\
      string[s].libs[string[s].liberties] = pos;\
    string[s].liberties++;\
    ml[pos] = liberty_mark;\
  } while (0)

#define ADD_NEIGHBOR(s, pos)\
  string[s].neighborlist[string[s].neighbors++] = string_number[pos]

#define DO_ADD_STONE(pos, color)\
  do {\
    PUSH_VERTEX(board[pos]);\
    board[pos] = color;\
    hashdata_invert_stone(&hashdata, pos, color);\
  } while (0)

#define DO_REMOVE_STONE(pos)\
  do {\
    PUSH_VERTEX(board[pos]);\
    hashdata_invert_stone(&hashdata, pos, board[pos]);\
    board[pos] = EMPTY;\
  } while (0)


/* ---------------------------------------------------------------- */


/* True if the data structures are up to date for the current 
 * board position.
 */
static int strings_initialized = 0;


/* Number of the next free string. */
static int next_string;


/* For marking purposes. */
static int ml[BOARDMAX];
static int liberty_mark;
static int string_mark;


/* Forward declarations. */
static int do_trymove(int pos, int color, int ignore_ko);
static void undo_trymove(void);
static void new_position(void);
static void init_board(void);
static int propagate_string(int stone, int str);
static void find_liberties_and_neighbors(int s);
static int do_remove_string(int s);
static void do_play_move(int pos, int color);
static int slow_approxlib(int pos, int color, int maxlib, int *libs);
static int incremental_sloppy_self_atari(int pos, int color);


/* Statistics. */
static int trymove_counter = 0;

/* Coordinates for the eight directions, ordered
 * south, west, north, east, southwest, northwest, northeast, southeast.
 */
int deltai[8] = { 1,  0, -1,  0,  1, -1, -1, 1};
int deltaj[8] = { 0, -1,  0,  1, -1, -1,  1, 1};
int delta[8]  = { NS, -1, -NS, 1, NS-1, -NS-1, -NS+1, NS+1};


/* ================================================================ */
/*                    Board initialization                          */
/* ================================================================ */

/*
 * Save board state.
 */

void
store_board(struct board_state *state)
{
  int k;

  gg_assert(stackp == 0);

  state->board_size = board_size;

  memcpy(state->board, board, sizeof(board));
  memcpy(state->initial_board, initial_board, sizeof(initial_board));

  state->board_ko_pos = board_ko_pos;
  state->white_captured = white_captured;
  state->black_captured = black_captured;
  
  state->initial_board_ko_pos = initial_board_ko_pos;
  state->initial_white_captured = initial_white_captured;
  state->initial_black_captured = initial_black_captured;
  
  state->move_history_pointer = move_history_pointer;
  for (k = 0; k < move_history_pointer; k++) {
    state->move_history_color[k] = move_history_color[k];
    state->move_history_pos[k] = move_history_pos[k];
  }

  state->komi = komi;
  state->move_number = movenum;
}


/*
 * Restore a saved board state.
 */

void
restore_board(struct board_state *state)
{
  int k;

  gg_assert(stackp == 0);

  board_size = state->board_size;

  memcpy(board, state->board, sizeof(board));
  memcpy(initial_board, state->initial_board, sizeof(initial_board));

  board_ko_pos = state->board_ko_pos;
  white_captured = state->white_captured;
  black_captured = state->black_captured;
  
  initial_board_ko_pos = state->initial_board_ko_pos;
  initial_white_captured = state->initial_white_captured;
  initial_black_captured = state->initial_black_captured;
  
  move_history_pointer = state->move_history_pointer;
  for (k = 0; k < move_history_pointer; k++) {
    move_history_color[k] = state->move_history_color[k];
    move_history_pos[k] = state->move_history_pos[k];
  }

  komi = state->komi;
  movenum = state->move_number;
  
  hashdata_recalc(&hashdata, board, board_ko_pos);
  new_position();
}


/*
 * Clear the internal board.
 */

void
clear_board(void)
{
  int k;

  gg_assert(board_size > 0 && board_size <= MAX_BOARD);
  
  memset(board, EMPTY, sizeof(board));
  memset(initial_board, EMPTY, sizeof(initial_board));
  for (k = 0; k < BOARDSIZE; k++) {
    if (!ON_BOARD2(I(k), J(k))) {
      board[k] = GRAY;
      initial_board[k] = GRAY;
    }
  }

  board_ko_pos = NO_MOVE;
  white_captured = 0;
  black_captured = 0;

  initial_board_ko_pos = NO_MOVE;
  initial_white_captured = 0;
  initial_black_captured = 0;

  move_history_pointer = 0;
  movenum = 0;
  
  hashdata_recalc(&hashdata, board, board_ko_pos);
  new_position();
}

/* Test the integrity of the gray border. */
int
test_gray_border(void)
{
  int k;

  gg_assert(board_size > 0 && board_size <= MAX_BOARD);
  
  for (k = 0; k < BOARDSIZE; k++)
    if (!ON_BOARD2(I(k), J(k)))
      if (board[k] != GRAY)
      	return k;
  
  return -1;
}


/* ================================================================ */
/*                      Temporary moves                             */
/* ================================================================ */


/* Stack of trial moves to get to current
 * position and which color made them. Perhaps 
 * this should be one array of a structure 
 */
static int stack[MAXSTACK];
static int move_color[MAXSTACK];

static Hash_data hashdata_stack[MAXSTACK];


/* KOMASTER_SCHEME 4 handles empty case differently from other schemes */
int
komaster_is_empty(int komaster, int kom_pos)
{
  if (KOMASTER_SCHEME == 4)
    return kom_pos == PASS_MOVE;
  else
    return komaster == EMPTY;
}

/* KOMASTER_SCHEME 4 handles komaster interpretation differently from 
 * other schemes */
const char *
komaster_to_string(int komaster, int kom_pos)
{
#if KOMASTER_SCHEME == 4 
  const char *b[4] = {"O-Illegal", "X-Illegal", "O-Legal", "X-Legal"};
  ASSERT1(komaster >= 0 && komaster <= 3, 0);
  if (kom_pos == PASS_MOVE) 
    return "EMPTY";
  return b[komaster];
#else
  UNUSED(kom_pos);
  return color_to_string(komaster);
#endif
}


/*
 * trymove pushes the position onto the stack, and makes a move
 * at pos of color. Returns one if the move is legal. The
 * stack pointer is only incremented if the move is legal.
 *
 * The way to use this is:
 *
 *   if (trymove(...)) {
 *      ...
 *      popgo();
 *   }   
 *
 * The message can be written as a comment to an sgf file using 
 * sgfdump(). str can be NO_MOVE if it is not needed but otherwise  
 * the location of str is included in the comment.
 */

int 
trymove(int pos, int color, const char *message, int str,
	int komaster, int kom_pos)
{
  /* Do the real work elsewhere. */
  if (!do_trymove(pos, color, 0))
    return 0;

  /* Store the move in an sgf tree if one is available. */
  if (sgf_dumptree) {
    char buf[100];

    if (message == NULL)
      message = "UNKNOWN";

    if (str == NO_MOVE) {
      if (!komaster_is_empty(komaster, kom_pos))
	gg_snprintf(buf, 100, "%s (variation %d, hash %lx, komaster %s:%s)", 
		    message, count_variations, hashdata.hashval[0],
		    komaster_to_string(komaster, kom_pos),
		    location_to_string(kom_pos));
      else
	gg_snprintf(buf, 100, "%s (variation %d, hash %lx)", 
		    message, count_variations, hashdata.hashval[0]);
    }
    else {
      if (!komaster_is_empty(komaster, kom_pos))
	gg_snprintf(buf, 100, 
		    "%s at %s (variation %d, hash %lx, komaster %s:%s)", 
		    message, location_to_string(str), count_variations,
		    hashdata.hashval[0], 
                    komaster_to_string(komaster, kom_pos),
		    location_to_string(kom_pos));
      else
	gg_snprintf(buf, 100, "%s at %s (variation %d, hash %lx)", 
		    message, location_to_string(str), count_variations,
		    hashdata.hashval[0]);
    }
    sgftreeAddPlayLast(sgf_dumptree, NULL, color, I(pos), J(pos));
    sgftreeAddComment(sgf_dumptree, NULL, buf);
  }
  
  if (count_variations)
    count_variations++;
  stats.nodes++;

  return 1;
}


/*
 * Special version for semeai reading.
 */

int 
semeai_trymove(int pos, int color, const char *message, int str1, int str2,
	       int owl_phase, int value)
{
  /* Do the real work elsewhere. */
  if (!do_trymove(pos, color, 0))
    return 0;

  if (message == NULL)
    message = "UNKNOWN";
  
  /* Store the move in an sgf tree if one is available. */
  if (sgf_dumptree) {
    char buf[100];
    char sbuf1[5], sbuf2[5];
    /* Simply calling location_to_string three times doesn't work
     * because the internal buffer of that function gets overwritten
     * too soon. So we allocate our own char buffers.
     */
    location_to_buffer(str1, sbuf1);
    location_to_buffer(str2, sbuf2);

    if (owl_phase)
      gg_snprintf(buf, 100, 
		  "%s in semeai %s-%s (variation %d, value %d, owl_phase)",
		  message, sbuf1, sbuf2, count_variations, value);
    else
      gg_snprintf(buf, 100, 
		  message, sbuf1, sbuf2, count_variations, value);
    sgftreeAddPlayLast(sgf_dumptree, NULL, color, I(pos), J(pos));
    sgftreeAddComment(sgf_dumptree, NULL, buf);
  }
  
  if (count_variations)
    count_variations++;
  stats.nodes++;

  return 1;
}


/*
 * tryko pushes the position onto the stack, and makes a move
 * at (pos) of (color). The move is allowed even if it is an
 * illegal ko capture. It is to be imagined that (color) has
 * made an intervening ko threat which was answered and now
 * the continuation is to be explored.
 *
 * Return 1 if the move is legal with the above caveat. Returns
 * zero if it is not legal because of suicide.
 */

int 
tryko(int pos, int color, const char *message, int komaster, int kom_pos)
{
  /* Do the real work elsewhere. */
  if (!do_trymove(pos, color, 1))
    return 0;

  if (sgf_dumptree) {
    char buf[100];
    if (message == NULL)
      message = "UNKNOWN";
    if (!komaster_is_empty(komaster, kom_pos))
      gg_snprintf(buf, 100, "tryko: %s (variation %d, %lx, komaster %s:%s)", 
		  message, count_variations, hashdata.hashval[0],
		  komaster_to_string(komaster, kom_pos), 
                  location_to_string(kom_pos));
    else
      gg_snprintf(buf, 100, "tryko: %s (variation %d, %lx)", 
		  message, count_variations, hashdata.hashval[0]);

    /* Add two pass moves to the SGF output to simulate the ko threat
     * and the answer.
     *
     * The reason we add these is that certain SGF viewers, including
     * Cgoban 1, won't properly display variations with illegal ko
     * captures. SGF FF[4] compliant browsers should have no problem
     * with this, though.
     */
    sgftreeAddPlayLast(sgf_dumptree, NULL, color, -1, -1);
    sgftreeAddComment(sgf_dumptree, NULL, "tenuki (ko threat)");
    sgftreeAddPlayLast(sgf_dumptree, NULL, OTHER_COLOR(color), -1, -1);
    sgftreeAddComment(sgf_dumptree, NULL, "tenuki (answers ko threat)");

    sgftreeAddPlayLast(sgf_dumptree, NULL, color, I(pos), J(pos));
    sgftreeAddComment(sgf_dumptree, NULL, buf);
  }
  
  if (count_variations)
    count_variations++;
  stats.nodes++;

  return 1;
}


/*
 * Do the main work of trymove() and tryko(), i.e. the common parts.
 * The ignore_ko flag tells whether an illegal ko capture may be done.
 * Return 1 if the move was valid, otherwise 0.
 */

static int 
do_trymove(int pos, int color, int ignore_ko)
{
  if (0 || KOMASTER_TRACE) {
    if (tracing_inside_komaster)
      gprintf("%o  ");

    if (ignore_ko)
      gprintf("%otryko");
    else
      gprintf("%otrymove");

    gprintf("%o %C %1m\n", color, pos);
  }

  /* 1. The move must be inside the board and the color must be BLACK
   * or WHITE.
   */
  ASSERT_ON_BOARD1(pos);
  gg_assert(color == BLACK || color == WHITE);
 
  /* Update the reading tree shadow. */
  shadow[pos] = 1;

  /* 2. The location must be empty. */
  if (board[pos] != EMPTY)
    return 0;

  /* 3. The location must not be the ko point, unless ignore_ko == 1. */
  if (!ignore_ko && pos == board_ko_pos) {
    if (board[WEST(pos)] == OTHER_COLOR(color)
	|| board[EAST(pos)] == OTHER_COLOR(color)) {
      RTRACE("%1m would violate the ko rule\n", pos);
      return 0;
    }
  }

  /* 4. Test for suicide. */
  if (is_suicide(pos, color)) {
    RTRACE("%1m would be suicide\n", pos);
    return 0;
  }
  
  /* Check for stack overflow. */
  if (stackp >= MAXSTACK-2) {
    fprintf(stderr, 
	    "gnugo: Truncating search. This is beyond my reading ability!\n");
    /* FIXME: Perhaps it's best to just assert here and be done with it? */
    if (0) {
      ASSERT1(0 && "trymove stack overflow", pos);
    }
    if (verbose > 0) {
      showboard(0);
      dump_stack();
    }
    fflush(stdout);
    fflush(stderr);
    return 0;
  }


  /* Only count trymove when we do create a new position. */
  trymove_counter++;
  
  /* So far, so good. Now push the move on the move stack. These are
   * needed for dump_stack().
   */
  stack[stackp] = pos;
  move_color[stackp] = color;

  /*
   * FIXME: Do we really have to store hashdata in a stack?
   *
   * Answer: No, we don't.  But for every stone that we add
   *         or remove, we must call hashdata_invert_stone(). This is
   *         not difficult per se, but the whole board.c 
   *         will have to be checked, and there is lots of room
   *         for mistakes.
   *
   *         At the same time, profiling shows that storing the
   *         hashdata in a stack doesn't take a lot of time, so
   *         this is not an urgent FIXME.
   */
  BEGIN_CHANGE_RECORD();
  PUSH_VALUE(board_ko_pos);
  memcpy(&hashdata_stack[stackp], &hashdata, sizeof(hashdata));

  if (board_ko_pos != NO_MOVE)
    hashdata_invert_ko(&hashdata, board_ko_pos);
  board_ko_pos = NO_MOVE;
  
  PUSH_VALUE(black_captured);
  PUSH_VALUE(white_captured);

  ++stackp;

  if (verbose == 4)
    dump_stack();

  do_play_move(pos, color);

  return 1;
}


/*
 * popgo pops the position from the stack.
 */

void
popgo()
{
  stackp--;
  
  undo_trymove();
  
  memcpy(&hashdata, &(hashdata_stack[stackp]), sizeof(hashdata));
  if (sgf_dumptree) {
    char buf[100];
    gg_snprintf(buf, 100, "(next variation: %d)", count_variations);
    sgftreeAddComment(sgf_dumptree, NULL, buf);
    sgf_dumptree->lastnode = sgf_dumptree->lastnode->parent;
    /* After tryko() we need to undo two pass nodes too. Since we have
     * no other way to identify ko moves, we skip all pass nodes.
     */
    while (is_pass_node(sgf_dumptree->lastnode, board_size))
      sgf_dumptree->lastnode = sgf_dumptree->lastnode->parent;
  }
}


/* Silent version of popgo(), suitable for use if you have called
 * do_trymove() without passing through trymove() or tryko().
 */

static void
silent_popgo(void)
{
  stackp--;
  undo_trymove();
  memcpy(&hashdata, &(hashdata_stack[stackp]), sizeof(hashdata));
}


/* Restore board state to the position before the last move. This is
 * accomplished by popping everything that was stored on the stacks
 * since the last BEGIN_CHANGE_RECORD().
 */

static void
undo_trymove()
{
  if (0 || KOMASTER_TRACE) {
    if (tracing_inside_komaster) {
      gprintf("%o  ");
    }
    gprintf("%opopgo\n");
  }

  gg_assert(strings_initialized);
  gg_assert(change_stack_pointer - change_stack <= STACK_SIZE);

  if (0) {
    gprintf("Change stack size = %d\n", change_stack_pointer - change_stack);
    gprintf("Vertex stack size = %d\n", vertex_stack_pointer - vertex_stack);
  }

  POP_MOVE();
  POP_VERTICES();
}




/*
 * dump_stack() for use under gdb prints the move stack. 
 */

void
dump_stack(void)
{
  int n;

  for (n = 0; n < stackp; n++)
    gprintf("%o%s:%1m ", move_color[n] == BLACK ? "B" : "W", stack[n]);
  
#if !TRACE_READ_RESULTS
  if (count_variations)
    gprintf("%o (variation %d)", count_variations-1);
#else
  gprintf("%o (%d)", hashdata.hashval[0]);
#endif

  gprintf("%o\n");
  fflush(stdout);
  fflush(stderr);
}


/* ================================================================ */
/*                     Permanent moves                              */
/* ================================================================ */


static void
reset_move_history(void)
{
  memcpy(initial_board, board, sizeof(board));
  initial_board_ko_pos = board_ko_pos;
  initial_white_captured = white_captured;
  initial_black_captured = black_captured;
  move_history_pointer = 0;
}

/* Place a stone on the board and update the hashdata. This operation
 * destroys all move history.
 */

void
add_stone(int pos, int color)
{
  ASSERT1(stackp == 0, pos);
  ASSERT_ON_BOARD1(pos);
  ASSERT1(board[pos] == EMPTY, pos);

  board[pos] = color;
  hashdata_invert_stone(&hashdata, pos, color);
  reset_move_history();
  new_position();
}


/* Remove a stone from the board and update the hashdata. This
 * operation destroys the move history.
 */

void
remove_stone(int pos)
{
  ASSERT1(stackp == 0, pos);
  ASSERT_ON_BOARD1(pos);
  ASSERT1(IS_STONE(board[pos]), pos);

  hashdata_invert_stone(&hashdata, pos, board[pos]);
  board[pos] = EMPTY;
  reset_move_history();
  new_position();
}

static void
play_move_no_history(int pos, int color)
{
#if CHECK_HASHING
  Hash_data oldkey;

  /* Check the hash table to see if it corresponds to the cumulative one. */
  hashdata_recalc(&oldkey, board, board_ko_pos);
#if FULL_POSITION_IN_HASH
  gg_assert(hashdata_diff_dump(&oldkey, &hashdata) == 0);
#else
  gg_assert(hashdata_compare(&oldkey, &hashdata) == 0);
#endif
#endif

  if (board_ko_pos != NO_MOVE)
    hashdata_invert_ko(&hashdata, board_ko_pos);
  board_ko_pos = NO_MOVE;

  /* If the move is a pass, we can skip some steps. */
  if (pos != PASS_MOVE) {
    ASSERT_ON_BOARD1(pos);
    ASSERT1(board[pos] == EMPTY, pos);

    /* Do play the move. */
    do_play_move(pos, color);

#if CHECK_HASHING
    /* Check the hash table to see if it equals the previous one. */
    hashdata_recalc(&oldkey, board, board_ko_pos);
#if FULL_POSITION_IN_HASH
    gg_assert(hashdata_diff_dump(&oldkey, &hashdata) == 0);
#else
    gg_assert(hashdata_compare(&oldkey, &hashdata) == 0);
#endif
#endif
  }
  new_position();
}

/* Load the initial position and replay the first n moves. */
static void
replay_move_history(int n)
{
  int k;
  
  memcpy(board, initial_board, sizeof(board));
  board_ko_pos = initial_board_ko_pos;
  white_captured = initial_white_captured;
  black_captured = initial_black_captured;
  new_position();

  for (k = 0; k < n; k++)
    play_move_no_history(move_history_pos[k], move_history_color[k]);
}

/* Play a move. If you want to test for legality you should first call
 * is_legal(). This function strictly follows the algorithm: 
 * 1. Place a stone of given color on the board.
 * 2. If there are any adjacent opponent strings without liberties,
 *    remove them and increase the prisoner count. 
 * 3. If the newly placed stone is part of a string without liberties,
 *    remove it and increase the prisoner count.
 *
 * In spite of the name "permanent move", this move can (usually) be
 * unplayed by undo_move(), but it is significantly more costly than
 * unplaying a temporary move. There are limitations on the available
 * move history, so under certain circumstances the move may not be
 * possible to unplay at a later time.
 */
void
play_move(int pos, int color)
{
  ASSERT1(stackp == 0, pos);

  if (move_history_pointer >= MAX_MOVE_HISTORY) {
    /* The move history is full. We resolve this by collapsing the
     * first about 10% of the moves into the initial position.
     */
    int number_collapsed_moves = 1 + MAX_MOVE_HISTORY / 10;
    int k;
    Intersection saved_board[BOARDSIZE];
    int saved_board_ko_pos = board_ko_pos;
    int saved_white_captured = white_captured;
    int saved_black_captured = black_captured;
    memcpy(saved_board, board, sizeof(board));

    replay_move_history(number_collapsed_moves);

    memcpy(initial_board, board, sizeof(board));
    initial_board_ko_pos = board_ko_pos;
    initial_white_captured = white_captured;
    initial_black_captured = black_captured;

    for (k = number_collapsed_moves; k < move_history_pointer; k++) {
      move_history_color[k - number_collapsed_moves] = move_history_color[k];
      move_history_pos[k - number_collapsed_moves] = move_history_pos[k];
    }
    move_history_pointer -= number_collapsed_moves;

    memcpy(board, saved_board, sizeof(board));
    board_ko_pos = saved_board_ko_pos;
    white_captured = saved_white_captured;
    black_captured = saved_black_captured;
  }

  move_history_color[move_history_pointer] = color;
  move_history_pos[move_history_pointer] = pos;
  move_history_pointer++;
  
  play_move_no_history(pos, color);
  
  movenum++;
}


/* Undo n permanent moves. Returns 1 if successful and 0 if it fails.
 * If n moves cannot be undone, no move is undone.
 */
int
undo_move(int n)
{
  gg_assert(stackp == 0);
  
  /* Fail if and only if the move history is too short. */
  if (move_history_pointer < n)
    return 0;

  replay_move_history(move_history_pointer - n);
  move_history_pointer -= n;
  movenum -= n;

  return 1;
}


/* Return the last move done by the opponent to color. Both if no move
 * was found or if the last move was a pass, PASS_MOVE is returned.
 */
int
get_last_opponent_move(int color)
{
  int k;
  
  for (k = move_history_pointer - 1; k >= 0; k--)
    if (move_history_color[k] == OTHER_COLOR(color))
      return move_history_pos[k];

  return PASS_MOVE;
}

/* Return the last move done by anyone. Both if no move was found or
 * if the last move was a pass, PASS_MOVE is returned.
 */
int
get_last_move()
{
  if (move_history_pointer == 0)
    return PASS_MOVE;

  return move_history_pos[move_history_pointer - 1];
}

/* ================================================================ */
/*                        Utility functions                         */
/* ================================================================ */


/*
 * Test if the move is a pass or not.  Return 1 if it is.
 */

int
is_pass(int pos)
{
  return pos == 0;
}


/*
 * is_legal(pos, color) determines whether the move (color) at
 * pos is legal.
 */

int 
is_legal(int pos, int color)
{
  /* 0. A pass move is always legal. */
  if (pos == 0)
    return 1;

  /* 1. The move must be inside the board. */
  ASSERT_ON_BOARD1(pos);

  /* 2. The location must be empty. */
  if (board[pos] != EMPTY) 
    return 0;

  /* 3. The location must not be the ko point. */
  if (pos == board_ko_pos)
    if (board[WEST(pos)] == OTHER_COLOR(color)
	|| board[EAST(pos)] == OTHER_COLOR(color)) {
      RTRACE("%1m would violate the ko rule\n", pos);
      return 0;
    }

  /* Check for stack overflow. */
  if (stackp >= MAXSTACK-2) {
    fprintf(stderr, 
	    "gnugo: Truncating search. This is beyond my reading ability!\n");
    /* FIXME: Perhaps it's best to just assert here and be done with it? */
    if (0) {
      ASSERT1(0 && "is_legal stack overflow", pos);
    }
    return 0;
  }

  /* Check for suicide. */
  if (!allow_suicide && is_suicide(pos, color)) {
    RTRACE("%1m would be suicide\n", pos);
    return 0;
  }
  
  return 1;
}


/*
 * is_suicide(pos, color) determines whether the move (color) at
 * (pos) would be a suicide.
 *
 * This is the case if
 * 1. There is no neighboring empty intersection.
 * 2. There is no neighboring opponent string with exactly one liberty.
 * 3. There is no neighboring friendly string with more than one liberty.
 */
int 
is_suicide(int pos, int color)
{
  ASSERT_ON_BOARD1(pos);
  ASSERT1(board[pos] == EMPTY, pos);

  if (!strings_initialized)
    init_board();
  
  /* Check for suicide. */
  if (LIBERTY(SOUTH(pos))
      || (ON_BOARD(SOUTH(pos))
	  && ((board[SOUTH(pos)] == color) ^ (LIBERTIES(SOUTH(pos)) == 1))))
    return 0;

  if (LIBERTY(WEST(pos))
      || (ON_BOARD(WEST(pos))
	  && ((board[WEST(pos)] == color) ^ (LIBERTIES(WEST(pos)) == 1))))
    return 0;

  if (LIBERTY(NORTH(pos))
      || (ON_BOARD(NORTH(pos))
	  && ((board[NORTH(pos)] == color) ^ (LIBERTIES(NORTH(pos)) == 1))))
    return 0;

  if (LIBERTY(EAST(pos))
      || (ON_BOARD(EAST(pos))
	  && ((board[EAST(pos)] == color) ^ (LIBERTIES(EAST(pos)) == 1))))
    return 0;

  return 1;
}


/*
 * is_illegal_ko_capture(pos, color) determines whether the move
 * (color) at (pos) would be an illegal ko capture.
 */
int 
is_illegal_ko_capture(int pos, int color)
{
  ASSERT_ON_BOARD1(pos);
  ASSERT1(board[pos] == EMPTY, pos);

  return (pos == board_ko_pos
	  && ((board[WEST(pos)] == OTHER_COLOR(color))
	      || (board[EAST(pos)] == OTHER_COLOR(color))));
}


/* Variation of trymove()/tryko() where ko captures (both conditional
 * and unconditional) must follow a komaster scheme.
 *
 * FIXME: This function could be optimized by integrating the
 * trymove()/tryko() code.
 */

#if KOMASTER_TRACE

/* This version of komaster_trymove is for tracing the calls to 
 * and return values from komaster_trymove.
 */
int komaster_trymove_ORIGINAL(int pos, int color,
			      const char *message, int str,
			      int komaster, int kom_pos,
			      int *new_komaster, int *new_kom_pos,
			      int *is_conditional_ko,
			      int consider_conditional_ko);

int
komaster_trymove(int pos, int color, const char *message, int str,
		 int komaster, int kom_pos,
		 int *new_komaster, int *new_kom_pos,
		 int *is_conditional_ko, int consider_conditional_ko)
{
  int ret;
  int traceon = 1;
  tracing_inside_komaster = 1;
  if (traceon) {
    gprintf("%okomaster_trymove %1m %C %s %1m %d\n",
                pos,
                color,
    	        komaster_to_string(komaster, kom_pos),
	        kom_pos,
                consider_conditional_ko);
  }
  ret = komaster_trymove_ORIGINAL(
                pos, color, message, str,
	  	komaster, kom_pos,
		new_komaster, new_kom_pos,
		is_conditional_ko, consider_conditional_ko);
  if (traceon) {
    if (*is_conditional_ko 
        || komaster != *new_komaster 
        || kom_pos  != *new_kom_pos) {
      char buff[255];
      gg_snprintf(buff, 255, "komaster:%s@%s->%s@%s is_conditional_ko:%d\n",
              komaster_to_string(komaster, kom_pos), 
              location_to_string(kom_pos),
              komaster_to_string(*new_komaster, *new_kom_pos),
              location_to_string(*new_kom_pos),
              *is_conditional_ko);
      gprintf("%o%s", buff);
      if (sgf_dumptree) {
        sgftreeAddComment(sgf_dumptree, NULL, buff);
      }
    }
  }
  tracing_inside_komaster = 0;
  return ret;
}


/* Replace komaster_trymove implementation with komaster_trymove_trace */
#define komaster_trymove komaster_trymove_ORIGINAL

#endif /* KOMASTER_TRACE */


#if KOMASTER_SCHEME == 1

/* I. Dan's simple scheme, O to move.
 * 
 * 1. Komaster is EMPTY.
 * 1a) Unconditional ko capture is allowed. Komaster remains EMPTY.
 * 1b) Conditional ko capture is allowed. Komaster is set to O and
 *     kom_pos to the location of the ko, where a stone was
 *     just removed.
 * 
 * 2. Komaster is O:
 * 2a) Only nested ko captures are allowed.
 * 2b) If komaster fills the ko at kom_pos then komaster reverts to
 *     EMPTY.
 * 
 * 3. Komaster is X:
 *    Play at kom_pos is not allowed. Any other ko capture
 *    is allowed. If O takes another ko, komaster becomes GRAY.
 * 
 * 4. Komaster is GRAY:
 *    Ko captures are not allowed. If the ko at kom_pos is
 *    filled then the komaster reverts to EMPTY.
 * 
 */
int
komaster_trymove(int pos, int color, const char *message, int str,
		 int komaster, int kom_pos,
		 int *new_komaster, int *new_kom_pos,
		 int *is_conditional_ko, int consider_conditional_ko)
{
  int other = OTHER_COLOR(color);
  int ko_move;
  int kpos;

  /* First we check whether the ko claimed by komaster has been
   * resolved. If that is the case, we revert komaster to EMPTY.
   *
   * The ko has been resolved in favor of the komaster if it has
   * been filled, or if it is no longer a ko and an opponent move
   * there is suicide. If komaster == GRAY we don't remember who
   * owns the ko so we have to try both colors.
   */
  if (komaster != EMPTY
      && (IS_STONE(board[kom_pos])
	  || (komaster != GRAY
	      && !is_ko(kom_pos, OTHER_COLOR(komaster), NULL)
	      && is_suicide(kom_pos, OTHER_COLOR(komaster)))
	  || (komaster == GRAY
	      && !is_ko(kom_pos, BLACK, NULL)
	      && !is_ko(kom_pos, WHITE, NULL)
	      && (is_suicide(kom_pos, BLACK)
		  || is_suicide(kom_pos, WHITE))))) {
    komaster = EMPTY;
    kom_pos = 0;
  }

  /* Usually the komaster parameters are unchanged. */
  *new_komaster = komaster;
  *new_kom_pos = kom_pos;

  *is_conditional_ko = 0;
  ko_move = is_ko(pos, color, &kpos);

  if (ko_move) {
    /* If opponent is komaster we may not capture his ko. */
    if (komaster == other && pos == kom_pos)
      return 0;

    /* If komaster is gray we may not capture ko at all. */
    if (komaster == GRAY)
      return 0;

    /* If we are komaster, we may only do nested captures. */
    if (komaster == color
	&& !(kpos == SW(kom_pos) || kpos == NW(kom_pos)
	     || kpos == NE(kom_pos) || kpos == SE(kom_pos)))
      return 0;
  }

  if (!trymove(pos, color, message, str, komaster, kom_pos)) {
    if (!consider_conditional_ko)
      return 0;

    if (!tryko(pos, color, message, komaster, kom_pos))
      return 0; /* Suicide. */
      
    *is_conditional_ko = 1;

    /* Conditional ko capture, set komaster parameters. */
    if (komaster == EMPTY) {
      *new_komaster = color;
      *new_kom_pos = kpos;
      return 1;
    }
  }

  if (!ko_move)
    return 1;

  if (komaster == other)
    *new_komaster = GRAY;
  *new_kom_pos = kpos;

  return 1;
}

#endif

#if KOMASTER_SCHEME == 5

#define GRAY_WHITE 4
#define GRAY_BLACK 5
#define WEAK_KO    6

/* V. Complex scheme, O to move.
 * 
 * 1. Komaster is EMPTY.
 * 1a) Unconditional ko capture is allowed.
 *       Komaster remains EMPTY if previous move was not a ko capture.
 *       Komaster is set to WEAK_KO if previous move was a ko capture
 *       and kom_pos is set to the old value of board_ko_pos.
 * 1b) Conditional ko capture is allowed. Komaster is set to O and
 *     kom_pos to the location of the ko, where a stone was
 *     just removed.
 * 
 * 2. Komaster is O:
 * 2a) Only nested ko captures are allowed. Kom_pos is moved to the
 *     new removed stone.
 * 2b) If komaster fills the ko at kom_pos then komaster reverts to
 *     EMPTY.
 * 
 * 3. Komaster is X:
 *    Play at kom_pos is not allowed. Any other ko capture
 *    is allowed. If O takes another ko, komaster becomes GRAY_X.
 * 
 * 4. Komaster is GRAY_O or GRAY_X:
 *    Ko captures are not allowed. If the ko at kom_pos is
 *    filled then the komaster reverts to EMPTY.
 *
 * 5. Komaster is WEAK_KO:
 * 5a) After a non-ko move komaster reverts to EMPTY.
 * 5b) Unconditional ko capture is only allowed if it is nested ko capture.
 *     Komaster is changed to WEAK_X and kom_pos to the old value of
 *     board_ko_pos.
 * 5c) Conditional ko capture is allowed according to the rules of 1b.
 */
int
komaster_trymove(int pos, int color, const char *message, int str,
		 int komaster, int kom_pos,
		 int *new_komaster, int *new_kom_pos,
		 int *is_conditional_ko, int consider_conditional_ko)
{
  int other = OTHER_COLOR(color);
  int ko_move;
  int kpos;
  int previous_board_ko_pos = board_ko_pos;

  /* First we check whether the ko claimed by komaster has been
   * resolved. If that is the case, we revert komaster to EMPTY.
   *
   * The ko has been resolved in favor of the komaster if it has
   * been filled, or if it is no longer a ko and an opponent move
   * there is suicide.
   */
  if (((komaster == WHITE || komaster == GRAY_WHITE)
       && (IS_STONE(board[kom_pos])
	   || (!is_ko(kom_pos, BLACK, NULL)
	       && is_suicide(kom_pos, BLACK))))
      || ((komaster == BLACK || komaster == GRAY_BLACK)
	  && (IS_STONE(board[kom_pos])
	      || (!is_ko(kom_pos, WHITE, NULL)
		  && is_suicide(kom_pos, WHITE))))) {
    komaster = EMPTY;
    kom_pos = NO_MOVE;
  }

  /* Usually the komaster parameters are unchanged. */
  *new_komaster = komaster;
  *new_kom_pos = kom_pos;

  *is_conditional_ko = 0;
  ko_move = is_ko(pos, color, &kpos);

  if (!ko_move) {
    if (komaster == WEAK_KO) {
      *new_komaster = EMPTY;
      *new_kom_pos = NO_MOVE;
    }
  }
  else {
    /* If opponent is komaster we may not capture his ko. */
    if (komaster == other && pos == kom_pos)
      return 0;

    /* If komaster is gray we may not capture ko at all. */
    if (komaster == GRAY_WHITE || komaster == GRAY_BLACK)
      return 0;

    /* If we are komaster, we may only do nested captures. */
    if (komaster == color
	&& !(kpos == SW(kom_pos) || kpos == NW(kom_pos)
	     || kpos == NE(kom_pos) || kpos == SE(kom_pos)))
      return 0;

    /* If komaster is WEAK_KO, we may only do nested ko capture or
     * conditional ko capture.
     */
    if (komaster == WEAK_KO) {
      if (pos != board_ko_pos
	  && !(kpos == SW(kom_pos) || kpos == NW(kom_pos)
	       || kpos == NE(kom_pos) || kpos == SE(kom_pos)))
	return 0;
    }
  }

  if (!trymove(pos, color, message, str, komaster, kom_pos)) {
    if (!consider_conditional_ko)
      return 0;

    if (!tryko(pos, color, message, komaster, kom_pos))
      return 0; /* Suicide. */
      
    *is_conditional_ko = 1;

    /* Conditional ko capture, set komaster parameters. */
    if (komaster == EMPTY || komaster == WEAK_KO) {
      *new_komaster = color;
      *new_kom_pos = kpos;
      return 1;
    }
  }

  if (!ko_move)
    return 1;

  if (komaster == other) {
    if (color == WHITE)
      *new_komaster = GRAY_BLACK;
    else
      *new_komaster = GRAY_WHITE;
  }
  else if (komaster == color) {
    /* This is where we update kom_pos after a nested capture. */
    *new_kom_pos = kpos;
  }
  else {
    /* We can reach here when komaster is EMPTY or WEAK_KO. If previous
     * move was also a ko capture, we now set komaster to WEAK_KO.
     */
    if (previous_board_ko_pos != NO_MOVE) {
      *new_komaster = WEAK_KO;
      *new_kom_pos = previous_board_ko_pos;
    }
  }
  
  return 1;
}

#endif

#if KOMASTER_SCHEME == 2

/* Simple komaster scheme, equivalent to the one implemented in 2.7.232.
 *
 * (II) Original 2.7.232 scheme, O to move.
 * 
 * 1. Komaster is EMPTY.
 * 1a) Unconditional ko capture is allowed. Komaster remains EMPTY.
 * 1b) Conditional ko capture is allowed. Komaster is set to O and
 *     kom_pos to the location of the ko, where a stone was
 *     just removed.
 * 
 * 2. Komaster is O:
 * 2a) Conditional ko capture is not allowed.
 * 2b) Unconditional ko capture is allowed. Komaster parameters unchanged.
 * 
 * 3. Komaster is X:
 * 3a) Conditional ko capture is not allowed.
 * 3b) Unconditional ko capture is allowed except for a move at kom_pos.
 *     Komaster parameters unchanged.
 * 
 * 4. Komaster is GRAY:
 *    Doesn't happen.
 */
int
komaster_trymove(int pos, int color, const char *message, int str,
		 int komaster, int kom_pos,
		 int *new_komaster, int *new_kom_pos,
		 int *is_conditional_ko, int consider_conditional_ko)
{
  int other = OTHER_COLOR(color);
  int kpos;

  /* Usually the komaster parameters are unchanged. */
  *new_komaster = komaster;
  *new_kom_pos = kom_pos;

  *is_conditional_ko = 0;

  /* If opponent is komaster we may not capture his ko. */
  if (is_ko(pos, color, &kpos) && komaster == other && pos == kom_pos)
    return 0;

  if (trymove(pos, color, message, str, komaster, kom_pos))
    return 1;

  /* Conditional ko captures are only allowed if the komaster is EMPTY. */
  if (!consider_conditional_ko || komaster != EMPTY)
    return 0;

  if (tryko(pos, color, message, komaster, kom_pos)) {
    /* Conditional ko capture, set komaster parameters. */
    *new_komaster = color;
    *new_kom_pos = kpos;
    *is_conditional_ko = 1;
    return 1;
  }
  
  /* If we come here, the move was a suicide. */
  return 0;
}

#endif

#if KOMASTER_SCHEME == 3

/* Slightly more complex komaster scheme.
 *
 * (III) Revised 2.7.232 version, O to move.
 * 
 * 1. Komaster is EMPTY.
 * 1a) Unconditional ko capture is allowed. Komaster remains EMPTY.
 * 1b) Conditional ko capture is allowed. Komaster is set to O and
 *     kom_pos to the location of the ko, where a stone was
 *     just removed.
 * 
 * 2. Komaster is O:
 *    Ko capture (both kinds) is allowed only if after playing the move,
 *    is_ko(kom_pos, X) returns false. In that case, kom_pos
 *    is updated to the new ko position, i.e. the stone captured by this
 *    move. 
 * 
 * 3. Komaster is X:
 * 3a) Conditional ko capture is not allowed.
 * 3b) Unconditional ko capture is allowed except for a move at kom_pos.
 *     Komaster parameters unchanged.
 * 
 * 4. Komaster is GRAY:
 *    Doesn't happen.
 * 
 */
int
komaster_trymove(int pos, int color, const char *message, int str,
		 int komaster, int kom_pos,
		 int *new_komaster, int *new_kom_pos,
		 int *is_conditional_ko, int consider_conditional_ko)
{
  int other = OTHER_COLOR(color);
  int ko_move;
  int kpos;

  /* Usually the komaster parameters are unchanged. */
  *new_komaster = komaster;
  *new_kom_pos = kom_pos;

  *is_conditional_ko = 0;

  /* If opponent is komaster we may not capture his ko. */
  ko_move = is_ko(pos, color, &kpos);
  if (ko_move && komaster == other && pos == kom_pos)
    return 0;

  if (!trymove(pos, color, message, str, komaster, kom_pos)) {
    /* Conditional ko captures are allowed if komaster is EMPTY or our
     * color.
     */
    if (!consider_conditional_ko || komaster == other)
      return 0;

    if (!tryko(pos, color, message, komaster, kom_pos))
      return 0; /* Suicide. */
      
    *is_conditional_ko = 1;

    /* Conditional ko capture, set komaster parameters. */
    if (komaster == EMPTY) {
      *new_komaster = color;
      *new_kom_pos = kpos;
    }
  }

  if (!ko_move)
    return 1;

  /* Remains to check that if we are komaster, the old ko is gone. */
  if (komaster != color)
    return 1;
  
  if (!is_ko(kom_pos, other, NULL)) {
    *new_kom_pos = kpos;
    return 1;
  }

  /* The old ko was still around, move not accepted. */
  popgo();
  return 0;
}

#endif


#if KOMASTER_SCHEME == 4

/* Returns true if the (pos1) and (pos2) are directly adjacent */
static int
is_next_to(int pos1, int pos2)
{
  int diff = pos1 - pos2;
  ASSERT_ON_BOARD1(pos1);
  ASSERT_ON_BOARD1(pos2);
  return diff == WE || diff == -WE || diff == NS || diff == -NS;
}

#define KOMASTER_X 1
#define KOMASTER_LEGAL 2

/* (IV) A different komaster approach.
 * As of 3.1.27 it's quite slow, and has been tested primarily
 * in the context of pattern-based reading, where the other 
 * komaster schemes weren't sufficient.  It does eliminate all
 * tested cyclic ko problems, but still fails a few regressions.
 *
 * Currently this komaster scheme is slow.
 * 
 * If kom_pos == PASS, then there is no ko restriction in place.
 * This allows us to forgo an empty komaster status.  The extra
 * 1/2 bit gives us room to track some more information.
 * Logically, the komaster can take these values:
 *   KOMASTER_O_legal    0
 *   KOMASTER_X_legal    1
 *   KOMASTER_O_illegal  2
 *   KOMASTER_X_illegal  3
 * These values aren't actually defined, but are built up out
 * of KOMASTER_X and KOMASTER_LEGAL.  They're used in the comments
 * below for simplicity, however.
 *
 * 1. Komaster_pos is PASS:
 *   Move always allowed.
 *   If it's a legal ko capture, then komaster becomes KOMASTER_{OX}_legal, 
 *     and komaster_pos set to (pos). OX determined by color parameter.
 *   If it's an illegal ko capture, komaster becomes KOMASTER_{OX}_illegal,
 *     and komaster_pos set to (pos). OX determined by color parameter.
 *     If !consider_conditional_ko, the move is not allowed.
 *   If it's not a ko, komaster_pos remains PASS.
 *
 * The treatment of KOMASTER_O and KOMASTER_X are always identical,
 * simply with the player to move switched.  Below, only one of the
 * two identical cases is given.
 *
 * 2. Komaster_pos is (pos), komaster is KOMASTER_O_legal:
 * 2a X to play:
 *    if pos takes same ko as komaster_pos:
 *      if legal, komaster becomes KOMASTER_X_illegal (!).
 *      if illegal (ko threat required),
 *        if consider_conditional_ko
 *          komaster becomes KOMASTER_X_illegal.
 *    X may make any other legal move, including kos, and the komaster
 *      does not change.
 * 2b O to play:
 *    O may resolve this ko, or take a related ko.
 *      if it's a related ko, the komaster_pos changes to this related ko.
 *    O may not take an unrelated ko.  He must resolve this one first.  This
 *      avoids triple ko problems.
 *
 * 3. Komaster_pos is (pos), komaster is KOMASTER_O_illegal:
 * 3a X simply resolves to a trymove(), since O's already made a conditional
 *      capture, X may not make a conditional capture too.
 * 3b O's play is identical to the legal case.  The next statment is probably 
 *      bogus: In particular, he may happily re-take this ko at any time, in 
 *      case X took it back (legally).
 */

int
komaster_trymove(int pos, int color, const char *message, int str,
		 int komaster, int kom_pos,
		 int *new_komaster, int *new_kom_pos,
		 int *is_conditional_ko, int consider_conditional_ko)
{
  int other = OTHER_COLOR(color);
  int ko_move;
  int kpos;
  int new_kom_color_flag = (color == BLACK) ? KOMASTER_X : 0;
  int kom_color = (komaster & KOMASTER_X) ? BLACK : WHITE;
  int log_illegal_moves = !!sgf_dumptree;

  if (!ON_BOARD1(kom_pos) 
      || board[kom_pos] != kom_color
      || !is_ko_point(kom_pos)
      || !board_ko_pos) {
    /* Try clearing komaster is the prev. move is not a ko.
     * i.e. it could have been an internal ko threat.*/
    /* ko's been resolved clear komaster */
    komaster = 0;
    kom_pos = PASS_MOVE;
  }
  /* komaster parameters are a typical default. */
  *new_komaster = komaster;
  *new_kom_pos = kom_pos;
  *is_conditional_ko = 0;

  ko_move = is_ko(pos, color, &kpos);

  if (kom_pos == PASS_MOVE) {
    if (trymove(pos, color, message, str, komaster, kom_pos)) {
      if (ko_move) {
        *new_kom_pos = pos;
        *new_komaster = KOMASTER_LEGAL + new_kom_color_flag;
      } /* else: legal non-ko move (usual case), no komaster change. */
      return 1;
    }
    else {
      if (consider_conditional_ko) {
        if (tryko(pos, color, message, komaster, kom_pos)) {
          /* Must be an illegal ko capture*/
          *new_kom_pos = pos;
          *new_komaster = new_kom_color_flag; /*don't set KOMASTER_LEGAL bit*/
          *is_conditional_ko = 1;
          return 1;
        }
	else {
          /* completely illegal move */
          return 0;
        }
      }
      else {
        /* don't even try to fake a ko threat. */
        if (log_illegal_moves) {
          if (tryko(pos, color, message, komaster, kom_pos)) {
            sgftreeAddComment(sgf_dumptree, NULL, 
                "ILLEGAL - Conditional ko capture not requested. (A)");
            popgo();
          }
        }
        return 0;
      }
    }
  }

  ASSERT_ON_BOARD1(kom_pos);

  /* Different color to move than komaster */
  if (new_kom_color_flag != (komaster & KOMASTER_X)) {
    /* if komaster is ILLEGAL, don't allow opponent to retake the ko. */
    if (!(komaster & KOMASTER_LEGAL)) {
      if (is_next_to(pos, kom_pos)) {
        /* don't let color retake an already spoken-for ko. */
        if (log_illegal_moves) {
          if (tryko(pos, color, message, komaster, kom_pos)) {
            sgftreeAddComment(sgf_dumptree, NULL, 
                "ILLEGAL - Can't retake other player's ko.");
            popgo();
          }
        }
        return 0;
      }
    }
    /* komaster is LEGAL */
    if (trymove(pos, color, message, str, komaster, kom_pos)) {
      if (!is_next_to(pos, kom_pos)) {
        /* An unrelated legal move (possibly another ko) */
        return 1;
      }
      else {
        /* legal ko re-capture */
        *new_kom_pos = pos;
        /* Might be worth trying to set the legal bit to see what happens. */
        /* Note we set the komaster to ILLEGAL anyway to avoid loops */
        *new_komaster = new_kom_color_flag;  /*don't set KOMASTER_LEGAL bit */
        *is_conditional_ko = 0;  /* Should this be set to 1? */
        return 1;
      }
    }
    else {
      if (consider_conditional_ko) {
        if (tryko(pos, color, message, komaster, kom_pos)) {
          /* illegal ko re-capture */
          *new_kom_pos = pos;
          *new_komaster = new_kom_color_flag;  /*don't set KOMASTER_LEGAL bit */
          *is_conditional_ko = 1;
          return 1;
        }
	else {
          /* completely illegal move */
          return 0;
        }
      }
      else {
        /* don't even try to fake a ko threat. */
        if (log_illegal_moves) {
          if (tryko(pos, color, message, komaster, kom_pos)) {
            sgftreeAddComment(sgf_dumptree, NULL, 
                "ILLEGAL - Conditional ko capture not requested. (B)");
            popgo();
          }
        }
        return 0;
      }
    }
  }

  /* komaster is trying to make a move */
  ASSERT1(new_kom_color_flag == (komaster & KOMASTER_X), pos);

  /* Same color case: same for LEGAL or ILLEGAL komaster status. */
  /* If we are komaster, we may only do nested captures. */
  if (ko_move
      && kpos != SW(kom_pos) 
      && kpos != NW(kom_pos)
      && kpos != NE(kom_pos)
      && kpos != SE(kom_pos)) {
    if (log_illegal_moves) {
      if (tryko(pos, color, message, komaster, kom_pos)) {
        sgftreeAddComment(sgf_dumptree, NULL, "ILLEGAL - Komaster can't switch kos.");
        popgo();
      }
    }
    return 0;
  }
  else
    return trymove(pos, color, message, str, komaster, kom_pos);

  /* Should have taken care of all cases by here. */
  gg_assert(0 && "Getting here should not be possible.");
}

#endif


#undef komaster_trymove  /* In case KOMASTER_TRACE munged it */


/* Determine whether vertex is on the edge. */
int
is_edge_vertex(int pos)
{
  ASSERT_ON_BOARD1(pos);
  if (!ON_BOARD(SW(pos))
      || !ON_BOARD(NE(pos)))
    return 1;

  return 0;
}



/* Count the number of liberties of the string at pos. pos must not be
 * empty.
 */
int
countlib(int str)
{
  ASSERT1(IS_STONE(board[str]), str);
  
  if (!strings_initialized)
    init_board();

  /* We already know the number of liberties. Just look it up. */
  return string[string_number[str]].liberties;
}


/* Find the liberties of the string at str. str must not be
 * empty. The locations of up to maxlib liberties are written into
 * libs[]. The full number of liberties is returned.
 *
 * If you want the locations of all liberties, whatever their number,
 * you should pass MAXLIBS as the value for maxlib and allocate space
 * for libs[] accordingly.
 */

int
findlib(int str, int maxlib, int *libs)
{
  int k;
  int liberties;
  int s;
  
  ASSERT1(IS_STONE(board[str]), str);
  ASSERT1(libs != NULL, str);
  
  if (!strings_initialized)
    init_board();

  /* We already have the list of liberties and only need to copy it to
   * libs[].
   *
   * However, if the string has more than MAX_LIBERTIES liberties the
   * list is truncated and if maxlib is also larger than MAX_LIBERTIES
   * we have to traverse the stones in the string in order to find
   * where the liberties are.
   */
  s = string_number[str];
  liberties = string[s].liberties;

  if (liberties <= MAX_LIBERTIES || maxlib <= MAX_LIBERTIES) {
    /* The easy case, it suffices to copy liberty locations from the
     * incrementally updated list.
     */
    for (k = 0; k < maxlib && k < liberties; k++)
      libs[k] = string[s].libs[k];
  }
  else {
    /* The harder case, where we have to traverse the stones in the
     * string. We don't have to check explicitly if we are back to
     * the start of the chain since we will run out of liberties
     * before that happens.
     */
    int pos;
    liberty_mark++;
    for (k = 0, pos = FIRST_STONE(s);
	 k < maxlib && k < liberties;
	 pos = NEXT_STONE(pos)) {
      if (UNMARKED_LIBERTY(SOUTH(pos))) {
	libs[k++] = SOUTH(pos);
	MARK_LIBERTY(SOUTH(pos));
	if (k >= maxlib)
	  break;
      }
      
      if (UNMARKED_LIBERTY(WEST(pos))) {
	libs[k++] = WEST(pos);
	MARK_LIBERTY(WEST(pos));
	if (k >= maxlib)
	  break;
      }
      
      if (UNMARKED_LIBERTY(NORTH(pos))) {
	libs[k++] = NORTH(pos);
	MARK_LIBERTY(NORTH(pos));
	if (k >= maxlib)
	  break;
      }
      
      if (UNMARKED_LIBERTY(EAST(pos))) {
	libs[k++] = EAST(pos);
	MARK_LIBERTY(EAST(pos));
	if (k >= maxlib)
	  break;
      }
    }
  }
      
  return liberties;
}

/* Count the liberties a stone of the given color would get if played
 * at (pos).  Captures are ignored based on the ignore_capture flag.
 * (pos) must be empty.  It will fail if there is more than one
 * string neighbor of the same color.  In this case, the return value
 * is -1.  Captures are not handled, so if ignore_capture is 0, and
 * a capture is required, -1 is returned.
 *
 * The intent of this function is to be as fast as possible, not
 * necessarily complete.
 *
 * Note well, that it relies on incremental data (?), and the number
 * of liberties (if over MAX_LIBERTIES) may be inaccurate (?).
 */

int
fastlib(int pos, int color, int ignore_capture)
{
  int k;
  int ally1 = NO_MOVE;
  int ally2 = NO_MOVE;
  int fast_liberties = 0;
  int other = OTHER_COLOR(color);

  ASSERT1(board[pos] == EMPTY, pos);
  ASSERT1(IS_STONE(color), pos);

  if (!strings_initialized)
    init_board();

  for (k = 0; k < 4; k++) {
    int neighbor = pos + delta[k];
    if (board[neighbor] == color) {
      if (ally1 != NO_MOVE) {
        if (string_number[ally1] != string_number[neighbor]) { 
          if (ally2 != NO_MOVE) {
            if (string_number[ally2] != string_number[neighbor]) {
	      /* More than 2 allies not implemented (yet).*/
              return -1;
            }
          }
	  else
            ally2 = neighbor;
        }
      }
      else
        ally1 = neighbor;
    }
  }
  for (k = 0; k < 4; k++) {
    int neighbor = pos + delta[k];
    int neighbor_color = board[neighbor];
    if (!ignore_capture
        && neighbor_color == other
	&& LIBERTIES(neighbor) == 1) {
      int neighbor_size = COUNTSTONES(neighbor);
      if ((neighbor_size <= 2 && !ally1)
          || (neighbor_size == 1
              && ally1 
              && !ally2
              && countstones(ally1) == 1)) {
        /* Here, we can gain only the adjacent new liberty. */
        fast_liberties++;
      }
      else
        return -1;
    }
    if (neighbor_color == EMPTY) {
      if ((!ally1 || !liberty_of_string(neighbor, ally1))
          && (!ally2 || !liberty_of_string(neighbor, ally2))) {
	fast_liberties++;
      }
    }
  }

  if (ally1)
    fast_liberties += LIBERTIES(ally1) - 1;
  if (ally2)
    fast_liberties += LIBERTIES(ally2) - count_common_libs(ally1, ally2);
  return fast_liberties;
}

/* Find the liberties a stone of the given color would get if played
 * at (pos), ignoring possible captures of opponent stones. (pos)
 * must be empty. If libs != NULL, the locations of up to maxlib
 * liberties are written into libs[]. The counting of
 * liberties may or may not be halted when maxlib is reached. The
 * number of liberties found is returned.
 *
 * If you want the number or the locations of all liberties, however
 * many they are, you should pass MAXLIBS as the value for maxlib and
 * allocate space for libs[] accordingly.
 */

int
approxlib(int pos, int color, int maxlib, int *libs)
{
  int k;
  int liberties = 0;
  int fast_liberties = 0;

  ASSERT1(board[pos] == EMPTY, pos);
  ASSERT1(IS_STONE(color), pos);

  if (!libs) {
    fast_liberties = fastlib(pos, color, 1);
    if (fast_liberties >= 0) {
      return fast_liberties;
    }
  }

  if (!strings_initialized)
    init_board();

  /* Look for empty neighbors and the liberties of the adjacent
   * strings of the given color. The algorithm below won't work
   * correctly if any of the adjacent strings have more than
   * MAX_LIBERTIES liberties AND maxlib is larger than MAX_LIBERTIES.
   * If this might be the case, we use a more robust fallback.
   */
  if (maxlib > MAX_LIBERTIES)
    return slow_approxlib(pos, color, maxlib, libs);
  
  /* Start by marking pos itself so it isn't counted among its own
   * liberties.
   */
  liberty_mark++;
  MARK_LIBERTY(pos);
    
  if (UNMARKED_LIBERTY(SOUTH(pos))) {
    if (libs != NULL && liberties < maxlib)
      libs[liberties] = SOUTH(pos);
    liberties++;
    /* Stop counting if we reach maxlib. */
    if (liberties >= maxlib)
      return liberties;
    MARK_LIBERTY(SOUTH(pos));
  }
  else if (board[SOUTH(pos)] == color) {
    int s = string_number[SOUTH(pos)];
    for (k = 0; k < string[s].liberties; k++) {
      int lib = string[s].libs[k];
      if (UNMARKED_LIBERTY(lib)) {
	if (libs != NULL && liberties < maxlib)
	  libs[liberties] = lib;
	liberties++;
	if (liberties >= maxlib)
	  return liberties;
	MARK_LIBERTY(lib);
      }
    }
  }
  
  if (UNMARKED_LIBERTY(WEST(pos))) {
    if (libs != NULL && liberties < maxlib)
      libs[liberties] = WEST(pos);
    liberties++;
    /* Stop counting if we reach maxlib. */
    if (liberties >= maxlib)
      return liberties;
    MARK_LIBERTY(WEST(pos));
  }
  else if (board[WEST(pos)] == color) {
    int s = string_number[WEST(pos)];
    for (k = 0; k < string[s].liberties; k++) {
      int lib = string[s].libs[k];
      if (UNMARKED_LIBERTY(lib)) {
	if (libs != NULL && liberties < maxlib)
	  libs[liberties] = lib;
	liberties++;
	if (liberties >= maxlib)
	  return liberties;
	MARK_LIBERTY(lib);
      }
    }
  }
  
  if (UNMARKED_LIBERTY(NORTH(pos))) {
    if (libs != NULL && liberties < maxlib)
      libs[liberties] = NORTH(pos);
    liberties++;
    /* Stop counting if we reach maxlib. */
    if (liberties >= maxlib)
      return liberties;
    MARK_LIBERTY(NORTH(pos));
  }
  else if (board[NORTH(pos)] == color) {
    int s = string_number[NORTH(pos)];
    for (k = 0; k < string[s].liberties; k++) {
      int lib = string[s].libs[k];
      if (UNMARKED_LIBERTY(lib)) {
	if (libs != NULL && liberties < maxlib)
	  libs[liberties] = lib;
	liberties++;
	if (liberties >= maxlib)
	  return liberties;
	MARK_LIBERTY(lib);
      }
    }
  }
  
  if (UNMARKED_LIBERTY(EAST(pos))) {
    if (libs != NULL && liberties < maxlib)
      libs[liberties] = EAST(pos);
    liberties++;
    /* Stop counting if we reach maxlib. */
    if (liberties >= maxlib)
      return liberties;
    /* Unneeded since we're about to leave. */
#if 0
    MARK_LIBERTY(EAST(pos));
#endif
  }
  else if (board[EAST(pos)] == color) {
    int s = string_number[EAST(pos)];
    for (k = 0; k < string[s].liberties; k++) {
      int lib = string[s].libs[k];
      if (UNMARKED_LIBERTY(lib)) {
	if (libs != NULL && liberties < maxlib)
	  libs[liberties] = lib;
	liberties++;
	if (liberties >= maxlib)
	  return liberties;
	MARK_LIBERTY(lib);
      }
    }
  }  

  return liberties;
}


/* Find the number of common liberties of the two strings at str1 and str2.
 */

int
count_common_libs(int str1, int str2)
{
  int libs1[MAXLIBS];
  int liberties1;
  int commonlibs;
  int k;
  
  ASSERT_ON_BOARD1(str1);
  ASSERT_ON_BOARD1(str2);
  ASSERT1(IS_STONE(board[str1]), str1);
  ASSERT1(IS_STONE(board[str2]), str2);
  
  if (!strings_initialized)
    init_board();

  if (countlib(str1) > countlib(str2)) {
    int tmp = str1;
    str1 = str2;
    str2 = tmp;
  }

  liberties1 = findlib(str1, MAXLIBS, libs1);
  commonlibs = 0;
  for (k = 0; k < liberties1; k++)
    if (neighbor_of_string(libs1[k], str2))
      commonlibs++;
  
  return commonlibs;
}


/* Find the common liberties of the two strings at str1 and str2. The
 * locations of up to maxlib common liberties are written into libs[].
 * The full number of common liberties is returned.
 *
 * If you want the locations of all common liberties, whatever their
 * number, you should pass MAXLIBS as the value for maxlib and
 * allocate space for libs[] accordingly.
 */

int
find_common_libs(int str1, int str2, int maxlib, int *libs)
{
  int libs1[MAXLIBS];
  int liberties1;
  int commonlibs;
  int k;
  
  ASSERT_ON_BOARD1(str1);
  ASSERT_ON_BOARD1(str2);
  ASSERT1(IS_STONE(board[str1]), str1);
  ASSERT1(IS_STONE(board[str2]), str2);
  ASSERT1(libs != NULL, str1);
  
  if (!strings_initialized)
    init_board();

  if (countlib(str1) > countlib(str2)) {
    int tmp = str1;
    str1 = str2;
    str2 = tmp;
  }

  liberties1 = findlib(str1, MAXLIBS, libs1);
  commonlibs = 0;
  for (k = 0; k < liberties1; k++) {
    if (neighbor_of_string(libs1[k], str2)) {
      if (commonlibs < maxlib)
	libs[commonlibs] = libs1[k];
      commonlibs++;
    }
  }
  
  return commonlibs;
}


/* Determine whether two strings have at least one common liberty.
 * If they have and lib != NULL, one common liberty is returned in *lib.
 */

int
have_common_lib(int str1, int str2, int *lib)
{
  int libs1[MAXLIBS];
  int liberties1;
  int k;
  
  ASSERT_ON_BOARD1(str1);
  ASSERT_ON_BOARD1(str2);
  ASSERT1(IS_STONE(board[str1]), str1);
  ASSERT1(IS_STONE(board[str2]), str2);
  
  if (!strings_initialized)
    init_board();

  if (countlib(str1) > countlib(str2)) {
    int tmp = str1;
    str1 = str2;
    str2 = tmp;
  }

  liberties1 = findlib(str1, MAXLIBS, libs1);
  for (k = 0; k < liberties1; k++) {
    if (neighbor_of_string(libs1[k], str2)) {
      if (lib)
	*lib = libs1[k];
      return 1;
    }
  }
  
  return 0;
}



/*
 * Report the number of stones in a string.
 */

int
countstones(int str)
{
  ASSERT_ON_BOARD1(str);
  ASSERT1(IS_STONE(board[str]), str);

  if (!strings_initialized)
    init_board();

  return COUNTSTONES(str);
}


/* Find the stones of the string at str. str must not be
 * empty. The locations of up to maxstones stones are written into
 * stones[]. The full number of stones is returned.
 */

int
findstones(int str, int maxstones, int *stones)
{
  int s;
  int size;
  int pos;
  int k;
  
  ASSERT_ON_BOARD1(str);
  ASSERT1(IS_STONE(board[str]), str);

  if (!strings_initialized)
    init_board();

  s = string_number[str];
  size = string[s].size;
  
  /* Traverse the stones of the string, by following the cyclic chain. */
  pos = FIRST_STONE(s);
  for (k = 0; k < maxstones && k < size; k++) {
    stones[k] = pos;
    pos = NEXT_STONE(pos);
  }

  return size;
}


/* chainlinks returns (in the (adj) array) the chains surrounding
 * the string at (str). The number of chains is returned.
 */

int 
chainlinks(int str, int adj[MAXCHAIN])
{
  struct string_data *s;
  int k;

  ASSERT1(IS_STONE(board[str]), str);

  if (!strings_initialized)
    init_board();

  /* We already have the list ready, just copy it and fill in the
   * desired information.
   */
  s = &string[string_number[str]];
  for (k = 0; k < s->neighbors; k++)
    adj[k] = string[s->neighborlist[k]].origin;

  return s->neighbors;
}


/* chainlinks2 returns (in adj array) the chains surrounding
 * the string at str, which has exactly lib liberties. The number
 * of such chains is returned.
 */

int
chainlinks2(int str, int adj[MAXCHAIN], int lib)
{
  struct string_data *s, *t;
  int k;
  int neighbors;

  ASSERT1(IS_STONE(board[str]), str);

  if (!strings_initialized)
    init_board();

  /* We already have the list ready, just copy the strings with the
   * right number of liberties.
   */
  neighbors = 0;
  s = &string[string_number[str]];
  for (k = 0; k < s->neighbors; k++) {
    t = &string[s->neighborlist[k]];
    if (t->liberties == lib)
      adj[neighbors++] = t->origin;
  }
  return neighbors;
}


/* chainlinks3 returns (in adj array) the chains surrounding
 * the string at str, which have less or equal lib liberties.
 * The number of such chains is returned.
 */

int
chainlinks3(int str, int adj[MAXCHAIN], int lib)
{
  struct string_data *s, *t;
  int k;
  int neighbors;

  ASSERT1(IS_STONE(board[str]), str);

  if (!strings_initialized)
    init_board();

  /* We already have the list ready, just copy the strings with the
   * right number of liberties.
   */
  neighbors = 0;
  s = &string[string_number[str]];
  for (k = 0; k < s->neighbors; k++) {
    t = &string[s->neighborlist[k]];
    if (t->liberties <= lib)
      adj[neighbors++] = t->origin;
  }
  return neighbors;
}


/*
 * Find the origin of a worm or a cavity, i.e. the point with the
 * smallest 1D board coordinate. The idea is to have a canonical
 * reference point for a string.
 */

int
find_origin(int str)
{
  ASSERT1(IS_STONE(board[str]), str);

  if (!strings_initialized)
    init_board();
  
  return string[string_number[str]].origin;
}


/* Determine whether a move by color at (pos) would be a self atari,
 * i.e. whether it would get more than one liberty. This function
 * returns true also for the case of a suicide move.
 */

int
is_self_atari(int pos, int color)
{
  int liberties;
  int result;
  
  ASSERT_ON_BOARD1(pos);
  ASSERT1(board[pos] == EMPTY, pos);
  ASSERT1(IS_STONE(color), pos);

  if (!strings_initialized)
    init_board();

  /* 1. Try first without really putting the stone on the board. */
  /* FIXME: Integrate incremental_sloppy_self_atari() here. */
  result = incremental_sloppy_self_atari(pos, color);
  if (result != -1)
    return result;

  /* 2. It was not so easy.  Now see if we can put the stone on the board.
   *    If we can't, this is a self atari.
   */
  if (!do_trymove(pos, color, 1))
    return 1;
  liberties = countlib(pos);
  silent_popgo();
  
  return liberties <= 1;
}


/*
 * Returns true if pos is a liberty of the string at str.
 */

int
liberty_of_string(int pos, int str)
{
  ASSERT_ON_BOARD1(pos);
  ASSERT_ON_BOARD1(str);
  if (IS_STONE(board[pos]))
    return 0;

  return neighbor_of_string(pos, str);
}


/*
 * Returns true if pos is a second order liberty of the string at str.
 */
int
second_order_liberty_of_string(int pos, int str)
{
  int k;
  ASSERT_ON_BOARD1(pos);
  ASSERT_ON_BOARD1(str);

  for (k = 0; k < 4; k++)
    if (board[pos + delta[k]] == EMPTY
	&& neighbor_of_string(pos + delta[k], str))
      return 1;

  return 0;
}


/*
 * Returns true if pos is adjacent to the string at str.
 */

int
neighbor_of_string(int pos, int str)
{
  int s;
  int color = board[str];

  ASSERT1(IS_STONE(color), str);
  ASSERT_ON_BOARD1(pos);

  if (!strings_initialized)
    init_board();

  s = string_number[str];
  
  if (board[SOUTH(pos)] == color
      && string_number[SOUTH(pos)] == s)
    return 1;

  if (board[WEST(pos)] == color
      && string_number[WEST(pos)] == s)
    return 1;

  if (board[NORTH(pos)] == color
      && string_number[NORTH(pos)] == s)
    return 1;

  if (board[EAST(pos)] == color
      && string_number[EAST(pos)] == s)
    return 1;

  return 0;
}


/*
 * Returns true if str1 and str2 belong to the same string.
 */

int
same_string(int str1, int str2)
{
  ASSERT_ON_BOARD1(str1);
  ASSERT_ON_BOARD1(str2);
  ASSERT1(IS_STONE(board[str1]), str1);
  ASSERT1(IS_STONE(board[str2]), str2);

  if (!strings_initialized)
    init_board();

  return string_number[str1] == string_number[str2];
}


/*
 * Returns true if the strings at str1 and str2 are adjacent.
 */

int
adjacent_strings(int str1, int str2)
{
  int s1, s2;
  int k;
  
  ASSERT_ON_BOARD1(str1);
  ASSERT_ON_BOARD1(str2);
  ASSERT1(IS_STONE(board[str1]), str1);
  ASSERT1(IS_STONE(board[str2]), str2);

  if (!strings_initialized)
    init_board();

  s1 = string_number[str1];
  s2 = string_number[str2];

  for (k = 0; k < string[s1].neighbors; k++)
    if (string[s1].neighborlist[k] == s2)
      return 1;

  return 0;
}


/*
 * Return true if the move (pos) by (color) is a ko capture
 * (whether capture is legal on this move or not). If so,
 * and if ko_pos is not a NULL pointer, then
 * *ko_pos returns the location of the captured ko stone.
 * If the move is not a ko capture, *ko_pos is set to 0.
 *
 * A move is a ko capture if and only if
 *    1. All neighbors are opponent stones.
 *    2. The number of captured stones is exactly one.
 */

int
is_ko(int pos, int color, int *ko_pos)
{
  int other = OTHER_COLOR(color);
  int captures = 0;
  int kpos = 0;
  
  ASSERT_ON_BOARD1(pos);
  ASSERT1(color == WHITE || color == BLACK, pos);

  if (!strings_initialized)
    init_board();
  
  if (ON_BOARD(SOUTH(pos))) {
    if (board[SOUTH(pos)] != other)
      return 0;
    else if (LIBERTIES(SOUTH(pos)) == 1) {
      kpos = SOUTH(pos);
      captures += string[string_number[SOUTH(pos)]].size;
      if (captures > 1)
	return 0;
    }
  }
  
  if (ON_BOARD(WEST(pos))) {
    if (board[WEST(pos)] != other)
      return 0;
    else if (LIBERTIES(WEST(pos)) == 1) {
      kpos = WEST(pos);
      captures += string[string_number[WEST(pos)]].size;
      if (captures > 1)
	return 0;
    }
  }
  
  if (ON_BOARD(NORTH(pos))) {
    if (board[NORTH(pos)] != other)
      return 0;
    else if (LIBERTIES(NORTH(pos)) == 1) {
      kpos = NORTH(pos);
      captures += string[string_number[NORTH(pos)]].size;
      if (captures > 1)
	return 0;
    }
  }
  
  if (ON_BOARD(EAST(pos))) {
    if (board[EAST(pos)] != other)
      return 0;
    else if (LIBERTIES(EAST(pos)) == 1) {
      kpos = EAST(pos);
      captures += string[string_number[EAST(pos)]].size;
      if (captures > 1)
	return 0;
    }
  }
  
  if (captures == 1) {
    if (ko_pos)
      *ko_pos = kpos;
    return 1;
  }
  return 0;
}


/* Return true if pos is either a stone, which if captured would give
 * ko, or if pos is an empty intersection adjacent to a ko stone.
 */
int
is_ko_point(int pos)
{
  ASSERT_ON_BOARD1(pos);

  if (!strings_initialized)
    init_board();
  
  if (board[pos] == EMPTY) {
    int color;
    if (ON_BOARD(SOUTH(pos)))
      color = board[SOUTH(pos)];
    else
      color = board[NORTH(pos)];
    if (IS_STONE(color) && is_ko(pos, OTHER_COLOR(color), NULL))
      return 1;
  }
  else {
    struct string_data *s = &string[string_number[pos]];
    if (s->liberties == 1 && s->size == 1
	&& is_ko(s->libs[0], OTHER_COLOR(s->color), NULL))
      return 1;
  }

  return 0;
}


/* Returns 1 if at least one string is captured when color plays at pos.
 */
int
does_capture_something(int pos, int color)
{
  int other = OTHER_COLOR(color);

  ASSERT1(board[pos] == EMPTY, pos);

  if (!strings_initialized)
    init_board();

  if (board[SOUTH(pos)] == other && LIBERTIES(SOUTH(pos)) == 1)
    return 1;
  
  if (board[WEST(pos)] == other && LIBERTIES(WEST(pos)) == 1)
    return 1;
  
  if (board[NORTH(pos)] == other && LIBERTIES(NORTH(pos)) == 1)
    return 1;
  
  if (board[EAST(pos)] == other && LIBERTIES(EAST(pos)) == 1)
    return 1;

  return 0;
}


/* For each stone in the string at pos, set mx to value mark. If
 * some of the stones in the string are marked prior to calling this
 * function, only the connected unmarked stones starting from pos
 * are guaranteed to become marked. The rest of the string may or may
 * not become marked. (In the current implementation, it will.)
 */
void
mark_string(int str, char mx[BOARDMAX], char mark)
{
  int pos = str;

  ASSERT1(IS_STONE(board[str]), str);

  if (!strings_initialized)
    init_board();

  do {
    mx[pos] = mark;
    pos = NEXT_STONE(pos);
  } while (pos != str);
}


/* Returns true if at least one move has been played at pos
 * at deeper than level 'cutoff' in the reading tree.
 */
int
move_in_stack(int pos, int cutoff)
{
  int k;
  for (k = cutoff; k < stackp; k++)
    if (stack[k] == pos)
      return 1;
  
  return 0;
}


/* Retrieve a move from the move stack. */
void
get_move_from_stack(int k, int *move, int *color)
{
  gg_assert(k < stackp);
  *move = stack[k];
  *color = move_color[k];
}

/* Return the number of stones of the indicated color(s) on the board.
 * This only counts stones in the permanent position, not stones placed
 * by trymove() or tryko(). Use stones_on_board(BLACK | WHITE) to get
 * the total number of stones on the board.
 */
int
stones_on_board(int color)
{
  static int stone_count_for_position = -1;
  static int white_stones = 0;
  static int black_stones = 0;

  gg_assert(stackp == 0);

  if (stone_count_for_position != position_number) {
    int pos;
    white_stones = 0;
    black_stones = 0;
    for (pos = BOARDMIN; pos < BOARDMAX; pos++) {
      if (board[pos] == WHITE)
	white_stones++;
      else if (board[pos] == BLACK)
	black_stones++;
    }
    
    stone_count_for_position = position_number;
  }

  return ((color & BLACK ? black_stones : 0) +
	  (color & WHITE ? white_stones : 0));
}


/* ===================== Statistics  ============================= */


/* Clear statistics. */
void
reset_trymove_counter()
{
  trymove_counter = 0;
}


/* Retrieve statistics. */
int
get_trymove_counter()
{
  return trymove_counter;
}


/* ================================================================ */
/*                      Lower level functions                       */
/* ================================================================ */


/* Don't trust the incremental string data until it's reinitialized.
 *
 * This function should be called if the board is modified by other
 * means than do_play_move() or undo_trymove().
 * It's also useful to force a recomputation of the strings if we
 * don't have any immediate plans to undo the move, because it recovers
 * undo stack space and holes in the 'string' array.
 */

/* We have reached a new position. Increase the position counter and
 * invalidate the incremental strings.
 */

static void
new_position(void)
{
  position_number++;
  strings_initialized = 0;
}


/* Set up incremental board structures and populate them with the
 * strings available in the position given by board[]. Clear the stacks
 * and start the mark numbers from zero. All undo information is lost
 * by calling this function.
 */

static void
init_board()
{
  int pos;
  int s;
  next_string = 0;
  liberty_mark = 0;
  string_mark = 0;
  CLEAR_STACKS();

  memset(string, 0, sizeof(string));
  memset(ml, 0, sizeof(ml));

  /* propagate_string relies on non-assigned stones to have
   * string_number -1.
   */
  for (pos = BOARDMIN; pos < BOARDMAX; pos++)
    if (ON_BOARD(pos))
      string_number[pos] = -1;

  /* Find the existing strings. */
  for (pos = BOARDMIN; pos < BOARDMAX; pos++) {
    if (!ON_BOARD(pos))
      continue;
    if (IS_STONE(board[pos]) && string_number[pos] == -1) {
      string_number[pos] = next_string;
      string[next_string].size = propagate_string(pos, pos);
      string[next_string].color = board[pos];
      string[next_string].origin = pos;
      string[next_string].mark = 0;
      next_string++;
    }
  }
  
  /* Fill in liberty and neighbor info. */
  for (s = 0; s < next_string; s++) {
    find_liberties_and_neighbors(s);
  }

  /* Now we can trust the information. */
  strings_initialized = 1;
}


#if 0

/*
 * Debug function. Dump all string information.
 */

static void
dump_incremental_board(void)
{
  int pos;
  int s;
  int i;
  
  for (pos = BOARDMIN; pos < BOARDMAX; pos++) {
    if (!ON_BOARD(pos))
      continue;
    if (board[pos] == EMPTY)
      fprintf(stderr, " . ");
    else
      fprintf(stderr, "%2d ", string_number[pos]);
    fprintf(stderr, "\n");
  }

  for (s = 0; s < next_string; s++) {
    if (board[string[s].origin] == EMPTY)
      continue;
    
    gprintf("%o%d %s %1m size %d, %d liberties, %d neighbors\n", s,
	    color_to_string(string[s].color),
	    string[s].origin, string[s].size,
	    string[s].liberties, string[s].neighbors);
    gprintf("%ostones:");

    pos = FIRST_STONE(s);
    do {
      gprintf("%o %1m", pos);
      pos = NEXT_STONE(pos);
    } while (!BACK_TO_FIRST_STONE(s, pos));
    
    gprintf("%o\nliberties:");
    for (i = 0; i < string[s].liberties; i++)
      gprintf("%o %1m", string[s].libs[i]);
    
    gprintf("%o\nneighbors:");
    for (i = 0; i < string[s].neighbors; i++)
      gprintf("%o %d(%1m)", string[s].neighborlist[i],
	      string[string[s].neighborlist[i]].origin);
    gprintf("%o\n\n");
  }
}
#endif


/* Build a string and its cyclic list representation from scratch.
 * propagate_string(stone, str) adds the stone (stone) to the string
 * (str) and recursively continues with not already included friendly
 * neighbors. To start a new string at (stone), use
 * propagate_string(stone, stone). The size of the string is returned.
 */

static int
propagate_string(int stone, int str)
{
  int size = 1;
  int k;
  
  if (stone == str) {
    /* Start a new string. */
    next_stone[stone] = stone;
  }
  else {
    /* Link the stone at (stone) to the string including (str) */
    string_number[stone] = string_number[str];
    next_stone[stone] = next_stone[str];
    next_stone[str] = stone;
  }

  /* Look in all four directions for more stones to add. */
  for (k = 0; k < 4; k++) {
    int d = delta[k];
    if (ON_BOARD(stone + d)
	&& board[stone + d] == board[stone]
	&& string_number[stone + d] == -1)
      size += propagate_string(stone + d, str);
  }
  
  return size;
}


/* Build the lists of liberties and neighbors of a string from
 * scratch. No information is pushed onto the stack by this function.
 */

static void
find_liberties_and_neighbors(int s)
{
  int pos;

  /* Clear the marks. */
  liberty_mark++;
  string_mark++;

  /* Traverse the stones of the string, by following the cyclic chain. */
  pos = FIRST_STONE(s);
  do {
    /* Look in each direction for new liberties or new neighbors. Mark
     * already visited liberties and neighbors.
     */
    if (UNMARKED_LIBERTY(SOUTH(pos))) {
      ADD_AND_MARK_LIBERTY(s, SOUTH(pos));
    }
    else if (UNMARKED_OPPONENT_STRING(s, SOUTH(pos))) {
      ADD_NEIGHBOR(s, SOUTH(pos));
      MARK_STRING(SOUTH(pos));
    }
    
    if (UNMARKED_LIBERTY(WEST(pos))) {
      ADD_AND_MARK_LIBERTY(s, WEST(pos));
    }
    else if (UNMARKED_OPPONENT_STRING(s, WEST(pos))) {
      ADD_NEIGHBOR(s, WEST(pos));
      MARK_STRING(WEST(pos));
    }
    
    if (UNMARKED_LIBERTY(NORTH(pos))) {
      ADD_AND_MARK_LIBERTY(s, NORTH(pos));
    }
    else if (UNMARKED_OPPONENT_STRING(s, NORTH(pos))) {
      ADD_NEIGHBOR(s, NORTH(pos));
      MARK_STRING(NORTH(pos));
    }
    
    if (UNMARKED_LIBERTY(EAST(pos))) {
      ADD_AND_MARK_LIBERTY(s, EAST(pos));
    }
    else if (UNMARKED_OPPONENT_STRING(s, EAST(pos))) {
      ADD_NEIGHBOR(s, EAST(pos));
      MARK_STRING(EAST(pos));
    }
    
    pos = NEXT_STONE(pos);
  } while (!BACK_TO_FIRST_STONE(s, pos));
}


/* Update the liberties of a string from scratch, first pushing the
 * old information.
 */

static void
update_liberties(int s)
{
  int pos;
  int k;

  /* Push the old information. */
  PUSH_VALUE(string[s].liberties);
  for (k = 0; k < string[s].liberties && k < MAX_LIBERTIES; k++) {
    PUSH_VALUE(string[s].libs[k]);
  }
  string[s].liberties = 0;

  /* Clear the liberty mark. */
  liberty_mark++;

  /* Traverse the stones of the string, by following the cyclic chain. */
  pos = FIRST_STONE(s);
  do {
    /* Look in each direction for new liberties. Mark already visited
     * liberties. 
     */
    if (UNMARKED_LIBERTY(SOUTH(pos))) {
      ADD_AND_MARK_LIBERTY(s, SOUTH(pos));
    }
    
    if (UNMARKED_LIBERTY(WEST(pos))) {
      ADD_AND_MARK_LIBERTY(s, WEST(pos));
    }
    
    if (UNMARKED_LIBERTY(NORTH(pos))) {
      ADD_AND_MARK_LIBERTY(s, NORTH(pos));
    }
    
    if (UNMARKED_LIBERTY(EAST(pos))) {
      ADD_AND_MARK_LIBERTY(s, EAST(pos));
    }
    
    pos = NEXT_STONE(pos);
  } while (!BACK_TO_FIRST_STONE(s, pos));
}


/* Remove a string from the list of neighbors and push the changed
 * information.
 */

static void
remove_neighbor(int str_number, int n)
{
  int k;
  int done = 0;
  struct string_data *s = &string[str_number];
  for (k = 0; k < s->neighbors; k++)
    if (s->neighborlist[k] == n) {
      /* We need to push the last entry too because it may become
       * destroyed later.
       */
      PUSH_VALUE(s->neighborlist[s->neighbors - 1]);
      PUSH_VALUE(s->neighborlist[k]);
      PUSH_VALUE(s->neighbors);
      s->neighborlist[k] = s->neighborlist[s->neighbors - 1];
      s->neighbors--;
      done = 1;
      break;
    }
  gg_assert(done);
}


/* Remove one liberty from the list of liberties, pushing changed
 * information. If the string had more liberties than the size of the
 * list, rebuild the list from scratch.
 */

static void
remove_liberty(int str_number, int pos)
{
  int k;
  struct string_data *s = &string[str_number];
  
  if (s->liberties > MAX_LIBERTIES)
    update_liberties(str_number);
  else {
    for (k = 0; k < s->liberties; k++)
      if (s->libs[k] == pos) {
	/* We need to push the last entry too because it may become
	 * destroyed later.
	 */
	PUSH_VALUE(s->libs[s->liberties - 1]);
	PUSH_VALUE(s->libs[k]);
	PUSH_VALUE(s->liberties);
	s->libs[k] = s->libs[s->liberties - 1];
	s->liberties--;
	break;
      }
  }
}


/* Remove a string from the board, pushing necessary information to
 * restore it. Return the number of removed stones.
 */

static int
do_remove_string(int s)
{
  int pos;
  int k;

  /* Traverse the stones of the string, by following the cyclic chain. */
  pos = FIRST_STONE(s);
  do {
    /* Push color, string number and cyclic chain pointers. */
    PUSH_VALUE(string_number[pos]);
    PUSH_VALUE(next_stone[pos]);
    DO_REMOVE_STONE(pos);
    pos = NEXT_STONE(pos);
  } while (!BACK_TO_FIRST_STONE(s, pos));

  /* The neighboring strings have obtained some new liberties and lost
   * a neighbor.
   */
  for (k = 0; k < string[s].neighbors; k++) {
    remove_neighbor(string[s].neighborlist[k], s);
    update_liberties(string[s].neighborlist[k]);
  }

  /* Update the number of captured stones. These are assumed to
   * already have been pushed.
   */
  if (string[s].color == WHITE)
    white_captured += string[s].size;
  else
    black_captured += string[s].size;
    
  return string[s].size;
}


/* We have played an isolated new stone and need to create a new
 * string for it.
 */
static void
create_new_string(int pos)
{
  int s;
  int color = board[pos];

  /* Get the next free string number. */
  PUSH_VALUE(next_string);
  s = next_string++;
  string_number[pos] = s;
  /* Set up a size one cycle for the string. */
  next_stone[pos] = pos;

  /* Set trivially known values and initialize the rest to zero. */
  string[s].color = color;
  string[s].size = 1;
  string[s].origin = pos;
  string[s].liberties = 0;
  string[s].neighbors = 0;
  string[s].mark = 0;

  /* Clear the string mark. */
  string_mark++;

  /* In each direction, look for a liberty or a nonmarked opponent
   * neighbor. Mark visited neighbors. There is no need to mark the
   * liberties since we can't find them twice. */
  if (LIBERTY(SOUTH(pos))) {
    ADD_LIBERTY(s, SOUTH(pos));
  }
  else if (UNMARKED_OPPONENT_STRING(s, SOUTH(pos))) {
    int s2 = string_number[SOUTH(pos)];
    /* Add the neighbor to our list. */
    ADD_NEIGHBOR(s, SOUTH(pos));
    /* Add us to our neighbor's list. */
    PUSH_VALUE(string[s2].neighbors);
    ADD_NEIGHBOR(s2, pos);
    MARK_STRING(SOUTH(pos));
  }
  
  if (LIBERTY(WEST(pos))) {
    ADD_LIBERTY(s, WEST(pos));
  }
  else if (UNMARKED_OPPONENT_STRING(s, WEST(pos))) {
    int s2 = string_number[WEST(pos)];
    /* Add the neighbor to our list. */
    ADD_NEIGHBOR(s, WEST(pos));
    /* Add us to our neighbor's list. */
    PUSH_VALUE(string[s2].neighbors);
    ADD_NEIGHBOR(s2, pos);
    MARK_STRING(WEST(pos));
  }
  
  if (LIBERTY(NORTH(pos))) {
    ADD_LIBERTY(s, NORTH(pos));
  }
  else if (UNMARKED_OPPONENT_STRING(s, NORTH(pos))) {
    int s2 = string_number[NORTH(pos)];
    /* Add the neighbor to our list. */
    ADD_NEIGHBOR(s, NORTH(pos));
    /* Add us to our neighbor's list. */
    PUSH_VALUE(string[s2].neighbors);
    ADD_NEIGHBOR(s2, pos);
    MARK_STRING(NORTH(pos));
  }
  
  if (LIBERTY(EAST(pos))) {
    ADD_LIBERTY(s, EAST(pos));
  }
  else if (UNMARKED_OPPONENT_STRING(s, EAST(pos))) {
    int s2 = string_number[EAST(pos)];
    /* Add the neighbor to our list. */
    ADD_NEIGHBOR(s, EAST(pos));
    /* Add us to our neighbor's list. */
    PUSH_VALUE(string[s2].neighbors);
    ADD_NEIGHBOR(s2, pos);
    /* No need to mark since no visits left. */
#if 0
    MARK_STRING(EAST(pos));
#endif
  }
}


/* We have played a stone with exactly one friendly neighbor. Add the
 * new stone to that string.
 */
static void
extend_neighbor_string(int pos, int s)
{
  int k;
  int liberties_updated = 0;

  /* Link in the stone in the cyclic list. */
  int pos2 = string[s].origin;
  next_stone[pos] = next_stone[pos2];
  PUSH_VALUE(next_stone[pos2]);
  next_stone[pos2] = pos;
  
  /* Do we need to update the origin? */
  if (pos < pos2) {
    PUSH_VALUE(string[s].origin);
    string[s].origin = pos;
  }
  
  string_number[pos] = s;

  /* The size of the string has increased by one. */
  PUSH_VALUE(string[s].size);
  string[s].size++;

  /* If s has too many liberties, we don't know where they all are and
   * can't update the liberties with the algorithm we otherwise
   * use. In that case we can only recompute the liberties from
   * scratch.
   */
  if (string[s].liberties > MAX_LIBERTIES) {
    update_liberties(s);
    liberties_updated = 1;
  }
  else {
    /* The place of the new stone is no longer a liberty. */
    remove_liberty(s, pos);
  }

  /* Mark old neighbors of the string. */
  string_mark++;
  for (k = 0; k < string[s].neighbors; k++)
    string[string[s].neighborlist[k]].mark = string_mark;

  /* Look at the neighbor locations of pos for new liberties and/or
   * neighbor strings.
   */

  /* If we find a liberty, look two steps away to determine whether
   * this already is a liberty of s.
   */
  if (LIBERTY(SOUTH(pos))) {
    if (!liberties_updated
	&& !(STRING_AT_VERTEX(SS(pos), s)
	     || STRING_AT_VERTEX(SW(pos), s)
	     || STRING_AT_VERTEX(SE(pos), s)))
      ADD_LIBERTY(s, SOUTH(pos));
  }
  else if (UNMARKED_OPPONENT_STRING(s, SOUTH(pos))) {
    int s2 = string_number[SOUTH(pos)];
    PUSH_VALUE(string[s].neighbors);
    ADD_NEIGHBOR(s, SOUTH(pos));
    PUSH_VALUE(string[s2].neighbors);
    ADD_NEIGHBOR(s2, pos);
    MARK_STRING(SOUTH(pos));
  }
  
  if (LIBERTY(WEST(pos))) {
    if (!liberties_updated
	&& !(STRING_AT_VERTEX(WW(pos), s)
	     || STRING_AT_VERTEX(NW(pos), s)
	     || STRING_AT_VERTEX(SW(pos), s)))
      ADD_LIBERTY(s, WEST(pos));
  }
  else if (UNMARKED_OPPONENT_STRING(s, WEST(pos))) {
    int s2 = string_number[WEST(pos)];
    PUSH_VALUE(string[s].neighbors);
    ADD_NEIGHBOR(s, WEST(pos));
    PUSH_VALUE(string[s2].neighbors);
    ADD_NEIGHBOR(s2, pos);
    MARK_STRING(WEST(pos));
  }
  
  if (LIBERTY(NORTH(pos))) {
    if (!liberties_updated
	&& !(STRING_AT_VERTEX(NN(pos), s)
	     || STRING_AT_VERTEX(NW(pos), s)
	     || STRING_AT_VERTEX(NE(pos), s)))
      ADD_LIBERTY(s, NORTH(pos));
  }
  else if (UNMARKED_OPPONENT_STRING(s, NORTH(pos))) {
    int s2 = string_number[NORTH(pos)];
    PUSH_VALUE(string[s].neighbors);
    ADD_NEIGHBOR(s, NORTH(pos));
    PUSH_VALUE(string[s2].neighbors);
    ADD_NEIGHBOR(s2, pos);
    MARK_STRING(NORTH(pos));
  }
  
  if (LIBERTY(EAST(pos))) {
    if (!liberties_updated
	&& !(STRING_AT_VERTEX(EE(pos), s)
	     || STRING_AT_VERTEX(NE(pos), s)
	     || STRING_AT_VERTEX(SE(pos), s)))
      ADD_LIBERTY(s, EAST(pos));
  }
  else if (UNMARKED_OPPONENT_STRING(s, EAST(pos))) {
    int s2 = string_number[EAST(pos)];
    PUSH_VALUE(string[s].neighbors);
    ADD_NEIGHBOR(s, EAST(pos));
    PUSH_VALUE(string[s2].neighbors);
    ADD_NEIGHBOR(s2, pos);
#if 0
    MARK_STRING(EAST(pos));
#endif
  }
  
}


/* Incorporate the string at pos with the string s.
 */

static void
assimilate_string(int s, int pos)
{
  int k;
  int last;
  int s2 = string_number[pos];
  string[s].size += string[s2].size;

  /* Walk through the s2 stones and change string number. Also pick up
   * the last stone in the cycle for later use.
   */
  pos = FIRST_STONE(s2);
  do {
    PUSH_VALUE(string_number[pos]);
    string_number[pos] = s;
    last = pos;
    pos = NEXT_STONE(pos);
  } while (!BACK_TO_FIRST_STONE(s2, pos));

  /* Link the two cycles together. */
  {
    int pos2 = string[s].origin;
    PUSH_VALUE(next_stone[last]);
    PUSH_VALUE(next_stone[pos2]);
    next_stone[last] = next_stone[pos2];
    next_stone[pos2] = string[s2].origin;
    
    /* Do we need to update the origin? */
    if (string[s2].origin < pos2)
      string[s].origin = string[s2].origin;
  }

  /* Pick up the liberties of s2 that we don't already have.
   * It is assumed that the liberties of s have been marked before
   * this function is called.
   */
  if (string[s2].liberties <= MAX_LIBERTIES) {
    for (k = 0; k < string[s2].liberties; k++) {
      int pos2 = string[s2].libs[k];
      if (UNMARKED_LIBERTY(pos2)) {
	ADD_AND_MARK_LIBERTY(s, pos2);
      }
    }
  }
  else {
    /* If s2 had too many liberties the above strategy wouldn't be
     * effective, since not all liberties are listed in
     * libs[] the chain of stones for s2 is no
     * longer available (it has already been merged with s) so we
     * can't reconstruct the s2 liberties. Instead we capitulate and
     * rebuild the list of liberties for s (including the neighbor
     * strings assimilated so far) from scratch.
     */
    liberty_mark++;          /* Reset the mark. */
    string[s].liberties = 0; /* To avoid pushing the current list. */
    update_liberties(s);
  }

  /* Remove s2 as neighbor to the neighbors of s2 and instead add s if
   * they don't already have added it. Also add the neighbors of s2 as
   * neighbors of s, unless they already have been added. The already
   * known neighbors of s are assumed to have been marked before this
   * function is called.
   */
  for (k = 0; k < string[s2].neighbors; k++) {
    int t = string[s2].neighborlist[k];
    remove_neighbor(t, s2);
    if (string[t].mark != string_mark) {
      PUSH_VALUE(string[t].neighbors);
      string[t].neighborlist[string[t].neighbors++] = s;
      string[s].neighborlist[string[s].neighbors++] = t;
      string[t].mark = string_mark;
    }
  }
}


/* Create a new string for the stone at pos and assimilate all
 * friendly neighbor strings.
 */

static void
assimilate_neighbor_strings(int pos)
{
  int s;
  int color = board[pos];

  /* Get the next free string number. */
  PUSH_VALUE(next_string);
  s = next_string++;
  PARANOID1(s < MAX_STRINGS, pos); 
  string_number[pos] = s;
  /* Set up a size one cycle for the string. */
  next_stone[pos] = pos;
  
  /* Set trivially known values and initialize the rest to zero. */
  string[s].color = color;
  string[s].size = 1;
  string[s].origin = pos;
  string[s].liberties = 0;
  string[s].neighbors = 0;

  /* Clear the marks. */
  liberty_mark++;
  string_mark++;

  /* Mark ourselves. */
  string[s].mark = string_mark;

  /* Look in each direction for
   *
   * 1. liberty: Add if not already visited.
   * 2. opponent string: Add it among our neighbors and us among its
   *    neighbors, unless already visited.
   * 3. friendly string: Assimilate.
   */
  if (UNMARKED_LIBERTY(SOUTH(pos))) {
    ADD_AND_MARK_LIBERTY(s, SOUTH(pos));
  }
  else if (UNMARKED_OPPONENT_STRING(s, SOUTH(pos))) {
    ADD_NEIGHBOR(s, SOUTH(pos));
    PUSH_VALUE(string[string_number[SOUTH(pos)]].neighbors);
    ADD_NEIGHBOR(string_number[SOUTH(pos)], pos);
    MARK_STRING(SOUTH(pos));
  }
  else if (UNMARKED_OWN_STRING(s, SOUTH(pos))) {
    assimilate_string(s, SOUTH(pos));
  }

  if (UNMARKED_LIBERTY(WEST(pos))) {
    ADD_AND_MARK_LIBERTY(s, WEST(pos));
  }
  else if (UNMARKED_OPPONENT_STRING(s, WEST(pos))) {
    ADD_NEIGHBOR(s, WEST(pos));
    PUSH_VALUE(string[string_number[WEST(pos)]].neighbors);
    ADD_NEIGHBOR(string_number[WEST(pos)], pos);
    MARK_STRING(WEST(pos));
  }
  else if (UNMARKED_OWN_STRING(s, WEST(pos))) {
    assimilate_string(s, WEST(pos));
  }
  
  if (UNMARKED_LIBERTY(NORTH(pos))) {
    ADD_AND_MARK_LIBERTY(s, NORTH(pos));
  }
  else if (UNMARKED_OPPONENT_STRING(s, NORTH(pos))) {
    ADD_NEIGHBOR(s, NORTH(pos));
    PUSH_VALUE(string[string_number[NORTH(pos)]].neighbors);
    ADD_NEIGHBOR(string_number[NORTH(pos)], pos);
    MARK_STRING(NORTH(pos));
  }
  else if (UNMARKED_OWN_STRING(s, NORTH(pos))) {
    assimilate_string(s, NORTH(pos));
  }
  
  if (UNMARKED_LIBERTY(EAST(pos))) {
    ADD_AND_MARK_LIBERTY(s, EAST(pos));
  }
  else if (UNMARKED_OPPONENT_STRING(s, EAST(pos))) {
    ADD_NEIGHBOR(s, EAST(pos));
    PUSH_VALUE(string[string_number[EAST(pos)]].neighbors);
    ADD_NEIGHBOR(string_number[EAST(pos)], pos);
    MARK_STRING(EAST(pos));
  }
  else if (UNMARKED_OWN_STRING(s, EAST(pos))) {
    assimilate_string(s, EAST(pos));
  }
}


/* Suicide at pos. Remove the neighboring friendly strings.
 */

static void
do_commit_suicide(int pos, int color)
{
  if (board[SOUTH(pos)] == color)
    do_remove_string(string_number[SOUTH(pos)]);

  if (board[WEST(pos)] == color)
    do_remove_string(string_number[WEST(pos)]);

  if (board[NORTH(pos)] == color)
    do_remove_string(string_number[NORTH(pos)]);

  if (board[EAST(pos)] == color)
    do_remove_string(string_number[EAST(pos)]);
}


/* Play a move without legality checking. Suicide is allowed.
 */

static void
do_play_move(int pos, int color)
{
  int other = OTHER_COLOR(color);
  int captured_stones = 0;
  int neighbor_allies = 0;
  int have_liberties = 0;
  int s = -1;
  
  if (!strings_initialized)
    init_board();
    
  /* Remove captured stones and check for suicide.*/
  if (board[SOUTH(pos)] == other && LIBERTIES(SOUTH(pos)) == 1)
    captured_stones += do_remove_string(string_number[SOUTH(pos)]);
  else if (LIBERTY(SOUTH(pos)) || (board[SOUTH(pos)] == color
				   && LIBERTIES(SOUTH(pos)) > 1))
    have_liberties = 1;

  if (board[WEST(pos)] == other && LIBERTIES(WEST(pos)) == 1)
    captured_stones += do_remove_string(string_number[WEST(pos)]);
  else if (LIBERTY(WEST(pos)) || (board[WEST(pos)] == color
				  && LIBERTIES(WEST(pos)) > 1))
    have_liberties = 1;

  if (board[NORTH(pos)] == other && LIBERTIES(NORTH(pos)) == 1)
    captured_stones += do_remove_string(string_number[NORTH(pos)]);
  else if (LIBERTY(NORTH(pos)) || (board[NORTH(pos)] == color
				   && LIBERTIES(NORTH(pos)) > 1))
    have_liberties = 1;

  if (board[EAST(pos)] == other && LIBERTIES(EAST(pos)) == 1)
    captured_stones += do_remove_string(string_number[EAST(pos)]);
  else if (LIBERTY(EAST(pos)) || (board[EAST(pos)] == color
				  && LIBERTIES(EAST(pos)) > 1))
    have_liberties = 1;

  /* No captures and no liberties -> suicide. */
  if (have_liberties == 0 && captured_stones == 0) {
    do_commit_suicide(pos, color);
    return;
  }
  
  /* Put down the stone. */
  DO_ADD_STONE(pos, color);

  /* Count the number of adjacent strings of my color and remove
   * pos as liberty for the adjacent opponent strings.
   */
  string_mark++;

  if (board[SOUTH(pos)] == color && UNMARKED_STRING(SOUTH(pos))) {
    neighbor_allies++;
    s = string_number[SOUTH(pos)];
    MARK_STRING(SOUTH(pos));
  }
  else if (board[SOUTH(pos)] == other && UNMARKED_STRING(SOUTH(pos))) {
    remove_liberty(string_number[SOUTH(pos)], pos);
    MARK_STRING(SOUTH(pos));
  }    
  
  if (board[WEST(pos)] == color && UNMARKED_STRING(WEST(pos))) {
    neighbor_allies++;
    s = string_number[WEST(pos)];
    MARK_STRING(WEST(pos));
  }
  else if (board[WEST(pos)] == other && UNMARKED_STRING(WEST(pos))) {
    remove_liberty(string_number[WEST(pos)], pos);
    MARK_STRING(WEST(pos));
  }    
  
  if (board[NORTH(pos)] == color && UNMARKED_STRING(NORTH(pos))) {
    neighbor_allies++;
    s = string_number[NORTH(pos)];
    MARK_STRING(NORTH(pos));
  }
  else if (board[NORTH(pos)] == other && UNMARKED_STRING(NORTH(pos))) {
    remove_liberty(string_number[NORTH(pos)], pos);
    MARK_STRING(NORTH(pos));
  }    
  
  if (board[EAST(pos)] == color && UNMARKED_STRING(EAST(pos))) {
    neighbor_allies++;
    s = string_number[EAST(pos)];
#if 0
    MARK_STRING(EAST(pos));
#endif
  }
  else if (board[EAST(pos)] == other && UNMARKED_STRING(EAST(pos))) {
    remove_liberty(string_number[EAST(pos)], pos);
#if 0
    MARK_STRING(EAST(pos));
#endif
  }    
  

  /* Choose strategy depending on the number of friendly neighbors. */
  if (neighbor_allies == 0)
    create_new_string(pos);
  else if (neighbor_allies == 1) {
    gg_assert(s >= 0);
    extend_neighbor_string(pos, s);
  }
  else
    assimilate_neighbor_strings(pos);

  /* Check whether this move was a ko capture and if so set 
   * board_ko_pos.
   *
   * No need to push board_ko_pos on the stack, 
   * because this has been done earlier.
   */
  s = string_number[pos];
  if (string[s].liberties == 1
      && string[s].size == 1
      && captured_stones == 1) {
    /* In case of a double ko: clear old ko position first. */
    if (board_ko_pos != NO_MOVE)
      hashdata_invert_ko(&hashdata, board_ko_pos);
    board_ko_pos = string[s].libs[0];
    hashdata_invert_ko(&hashdata, board_ko_pos);
  }
}


/* Find the liberties a move of the given color at pos would have,
 * excluding possible captures, by traversing all adjacent friendly
 * strings. This is a fallback used by approxlib() when a
 * faster algorithm can't be used.
 */

static int
slow_approxlib(int pos, int color, int maxlib, int *libs)
{
  int liberties = 0;
  int k;

  liberty_mark++;
  MARK_LIBERTY(pos);
  string_mark++;
  for (k = 0; k < 4; k++) {
    int d = delta[k];
    if (UNMARKED_LIBERTY(pos + d)) {
      if (libs)
	libs[liberties] = pos + d;
      liberties++;
      if (liberties == maxlib)
	return liberties;
      MARK_LIBERTY(pos + d);
    }
    else if (board[pos + d] == color
	     && UNMARKED_STRING(pos + d)) {
      int s = string_number[pos + d];
      int pos2;
      pos2 = FIRST_STONE(s);
      do {
	int l;
	for (l = 0; l < 4; l++) {
	  int d2 = delta[l];
	  if (UNMARKED_LIBERTY(pos2 + d2)) {
	    if (libs)
	      libs[liberties] = pos2 + d2;
	    liberties++;
	    if (liberties == maxlib)
	      return liberties;
	    MARK_LIBERTY(pos2 + d2);
	  }
	}
	
	pos2 = NEXT_STONE(pos2);
      } while (!BACK_TO_FIRST_STONE(s, pos2));
      MARK_STRING(pos + d);
    }
  }
  return liberties;
}


/* Determine whether a move by color at pos might be a self atari.
 * This function is sloppy in that it only does a quick check for two
 * liberties and might miss certain cases.
 * Return value 0 means it cannot be a self atari.
 * Return value 1 means it definitely is a self atari.
 * Return value -1 means uncertain.
 */

static int
incremental_sloppy_self_atari(int pos, int color)
{
  int other = OTHER_COLOR(color);
  /* number of empty neighbors */
  int trivial_liberties = 0;
  /* number of captured opponent strings */
  int captures = 0;
  /* Whether there is a friendly neighbor with a spare liberty. If it
   * has more than one spare liberty we immediately return 0.
   */
  int far_liberties = 0;

  /* Clear string mark. */
  string_mark++;
  
  if (board[SOUTH(pos)] == EMPTY)
    trivial_liberties++;
  else if (board[SOUTH(pos)] == color) {
    if (LIBERTIES(SOUTH(pos)) > 2)
      return 0;
    if (LIBERTIES(SOUTH(pos)) == 2)
      far_liberties++;
  }
  else if (board[SOUTH(pos)] == other
	   && LIBERTIES(SOUTH(pos)) == 1 && UNMARKED_STRING(SOUTH(pos))) {
    captures++;
    MARK_STRING(SOUTH(pos));
  }

  if (board[WEST(pos)] == EMPTY)
    trivial_liberties++;
  else if (board[WEST(pos)] == color) {
    if (LIBERTIES(WEST(pos)) > 2)
      return 0;
    if (LIBERTIES(WEST(pos)) == 2)
      far_liberties++;
  }
  else if (board[WEST(pos)] == other
	   && LIBERTIES(WEST(pos)) == 1 && UNMARKED_STRING(WEST(pos))) {
    captures++;
    MARK_STRING(WEST(pos));
  }

  if (board[NORTH(pos)] == EMPTY)
    trivial_liberties++;
  else if (board[NORTH(pos)] == color) {
    if (LIBERTIES(NORTH(pos)) > 2)
      return 0;
    if (LIBERTIES(NORTH(pos)) == 2)
      far_liberties++;
  }
  else if (board[NORTH(pos)] == other
	   && LIBERTIES(NORTH(pos)) == 1 && UNMARKED_STRING(NORTH(pos))) {
    captures++;
    MARK_STRING(NORTH(pos));
  }

  if (board[EAST(pos)] == EMPTY)
    trivial_liberties++;
  else if (board[EAST(pos)] == color) {
    if (LIBERTIES(EAST(pos)) > 2)
      return 0;
    if (LIBERTIES(EAST(pos)) == 2)
      far_liberties++;
  }
  else if (board[EAST(pos)] == other
	   && LIBERTIES(EAST(pos)) == 1 && UNMARKED_STRING(EAST(pos))) {
    captures++;
    MARK_STRING(EAST(pos));
  }

  /* Each captured string is guaranteed to produce at least one
   * liberty. These are disjoint from both trivial liberties and far
   * liberties. The two latter may however coincide.
   */
  
  if (trivial_liberties + captures >= 2)
    return 0;

  if ((far_liberties > 0) + captures >= 2)
    return 0;

  if (captures == 0 && far_liberties + trivial_liberties <= 1)
    return 1;

  return -1;
}


/* ================================================================ *
 * The following functions don't actually belong here. They are  
 * only here because they are faster here where they have access to
 * the incremental data structures. 
 * ================================================================ */


/* Help collect the data needed by order_moves() in reading.c.
 * It's the caller's responsibility to initialize the result parameters.
 */
#define NO_UNROLL 0
void
incremental_order_moves(int move, int color, int str,
			int *number_edges, int *number_same_string,
			int *number_own, int *number_opponent,
			int *captured_stones, int *threatened_stones,
			int *saved_stones, int *number_open)
{
#if NO_UNROLL == 1
  int pos;
  int k;

  /* Clear the string mark. */
  string_mark++;

  for (k = 0; k < 4; k++) {
    pos = move + delta[k];
    if (!ON_BOARD(pos))
      (*number_edges)++;
    else if (board[pos] == EMPTY)
      (*number_open)++;
    else {
      int s = string_number[pos];
      if (string_number[str] == s)
	(*number_same_string)++;
      
      if (board[pos] == color) {
	(*number_own)++;
	if (string[s].liberties == 1)
	  (*saved_stones) += string[s].size;
      }
      else {
	(*number_opponent)++;
	if (string[s].liberties == 1) {
	  int r;
	  struct string_data *t;
	  (*captured_stones) += string[s].size;
	  for (r = 0; r < string[s].neighbors; r++) {
	    t = &string[string[s].neighborlist[r]];
	    if (t->liberties == 1)
	      (*saved_stones) += t->size;
	  }
	}
	else if (string[s].liberties == 2 && UNMARKED_STRING(pos)) {
	  (*threatened_stones) += string[s].size;
	  MARK_STRING(pos);
	}
      }
    }
  }
  
#else
#define code1(arg) \
  if (!ON_BOARD(arg)) \
    (*number_edges)++; \
  else if (board[arg] == EMPTY) \
    (*number_open)++; \
  else { \
    int s = string_number[arg]; \
    if (string_number[str] == s) \
      (*number_same_string)++; \
    if (board[arg] == color) { \
      (*number_own)++; \
      if (string[s].liberties == 1) \
	(*saved_stones) += string[s].size; \
    } \
    else { \
      (*number_opponent)++; \
      if (string[s].liberties == 1) { \
	int r; \
	struct string_data *t; \
	(*captured_stones) += string[s].size; \
	for (r = 0; r < string[s].neighbors; r++) { \
	  t = &string[string[s].neighborlist[r]]; \
	  if (t->liberties == 1) \
	    (*saved_stones) += t->size; \
	} \
      } \
      else if (string[s].liberties == 2 && UNMARKED_STRING(arg)) { \
	(*threatened_stones) += string[s].size; \
        MARK_STRING(arg); \
      } \
    } \
  }

  /* Clear the string mark. */
  string_mark++;

  code1(SOUTH(move));
  code1(WEST(move));
  code1(NORTH(move));
  code1(EAST(move));
#endif
}


/* Reorientation of point (i, j) into (*ri, *rj). */
void
rotate2(int i, int j, int *ri, int *rj, int rot)
{
  ASSERT2(rot >= 0 && rot < 8, i, j);
  if (is_pass(POS(i, j))) {
    *ri = i;
    *rj = j;
    return;
  }
  ASSERT_ON_BOARD2(i, j);
  rotate(i, j, ri, rj, board_size, rot);
}

/* Inverse reorientation of reorientation rot. */
void
inv_rotate2(int i, int j, int *ri, int *rj, int rot)
{
  ASSERT2(rot >= 0 && rot < 8, i, j);
  if (is_pass(POS(i, j))) {
    *ri = i;
    *rj = j;
    return;
  }
  ASSERT_ON_BOARD2(i, j);
  inv_rotate(i, j, ri, rj, board_size, rot);
}

/* 1D board: return reorientation of point pos */
int
rotate1(int pos, int rot)
{
  int i, j;
  ASSERT1(rot >= 0 && rot < 8, pos);
  if (is_pass(pos))
    return PASS_MOVE;
  ASSERT_ON_BOARD1(pos);
  rotate2(I(pos), J(pos), &i, &j, rot);
  return POS(i, j);
}

/* 1D board: return inverse reorientation of point pos */
int
inv_rotate1(int pos, int rot)
{
  int i, j;
  ASSERT1(rot >= 0 && rot < 8, pos);
  if (is_pass(pos))
    return PASS_MOVE;
  ASSERT_ON_BOARD1(pos);
  inv_rotate2(I(pos), J(pos), &i, &j, rot);
  return POS(i, j);
}


/*
 * Local Variables:
 * tab-width: 8
 * c-basic-offset: 2
 * End:
 */
