/*
 * file_info.cpp
 *
 * Copyright (C) 2003 Ed Cogburn <ecogburn@xtn.net>
 *
 * This file is part of the program "d: The Directory Lister".
 *
 * "d: The Directory Lister" is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any later version.
 *
 * "d: The Directory Lister" is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with "d: The Directory Lister" in a file named "COPYING".
 * If not, visit their website at http://www.gnu.org for a copy, or request a copy by writing to:
 *     Free Software Foundation, Inc.
 *     59 Temple Place, Suite 330
 *     Boston, MA 02111-1307 USA
*/



#include "globals.hpp"
#include <sys/stat.h>
#include <sys/types.h>
#include <grp.h>
#include <pwd.h>
#include <unistd.h>



//-----------------------------------------------------------------------------
bool file_info::operator< (file_info& rhs)
{
	return ( name.full() < rhs.name.full() );
}



//-----------------------------------------------------------------------------
bool file_info::operator> (file_info& rhs)
{
	return ( name.full() > rhs.name.full() );
}



//-----------------------------------------------------------------------------
bool file_info::operator== (file_info& rhs)
{
	return ( name.full() == rhs.name.full() );
}



//-----------------------------------------------------------------------------
void file_info::read_file_info()
{
	struct stat st;
	struct passwd *pw;
	struct group *gr;

	if (lstat(name.full().c_str(), &st) == 0) {
		exists = true;

		// Get flags
		if (S_ISDIR(st.st_mode))
			is_dir = true;
		if (S_ISCHR(st.st_mode))
			is_char_device = true;
		if (S_ISBLK(st.st_mode))
			is_block_device = true;
		if (S_ISREG(st.st_mode))
			is_regular_file = true;
		if (S_ISFIFO(st.st_mode))
			is_pipe = true;
		if (S_ISLNK(st.st_mode))
			is_symlink = true;
		if (S_ISSOCK(st.st_mode))
			is_socket = true;
		if (st.st_mode & S_IRUSR)
			perm.owner_r = true;
		if (st.st_mode & S_IWUSR)
			perm.owner_w = true;
		if (st.st_mode & S_IXUSR)
			perm.owner_x = true;
		if (st.st_mode & S_IRGRP)
			perm.group_r = true;
		if (st.st_mode & S_IWGRP)
			perm.group_w = true;
		if (st.st_mode & S_IXGRP)
			perm.group_x = true;
		if (st.st_mode & S_IROTH)
			perm.other_r = true;
		if (st.st_mode & S_IWOTH)
			perm.other_w = true;
		if (st.st_mode & S_IXOTH)
			perm.other_x = true;
		if (st.st_mode & S_ISUID)
			perm.suid_x = true;
		if (st.st_mode & S_ISGID)
			perm.sgid_x = true;
		if (st.st_mode & S_ISVTX)
			perm.sticky = true;

		serial_nbr = st.st_ino;
		dev_nbr = st.st_dev;
		rdev_nbr = st.st_rdev;
		hardlinks = st.st_nlink;
		size = st.st_size;
		access_time = st.st_atime;
		mod_time = st.st_mtime;
		attrib_time = st.st_ctime;

		uid = st.st_uid;
		gid = st.st_gid;

		pw = getpwuid(uid);
		if (pw)
			uid_name = pw->pw_name;
		else
			uid_name = "unknown";

		gr = getgrgid(gid);
		if (gr)
			gid_name = gr->gr_name;
		else
			gid_name = "unknown";

		if (is_symlink) {
			char buf[PATH_MAX];

			int len = readlink(name.full().c_str(), buf, PATH_MAX);
			if (len == PATH_MAX)
				len--;
			if (len > 0) {
				buf[len] = '\0';
				symlink_name = buf;
			}
			if (stat(name.full().c_str(), &st) != 0)
				is_dangling_symlink = true;
		}
	}
}



//-----------------------------------------------------------------------------
void file_info::init()
{
	exists = false;

	is_dir = false;
	is_char_device = false;
	is_block_device = false;
	is_regular_file = false;
	is_pipe = false;
	is_symlink = false;
	is_socket = false;

	is_dangling_symlink = false;
	symlink_name = "";

	perm.owner_r = false;
	perm.owner_w = false;
	perm.owner_x = false;
	perm.group_r = false;
	perm.group_w = false;
	perm.group_x = false;
	perm.other_r = false;
	perm.other_w = false;
	perm.other_x = false;
	perm.suid_x = false;
	perm.sgid_x = false;
	perm.sticky = false;

	serial_nbr = 0;
	dev_nbr = 0;
	rdev_nbr = 0;
	hardlinks = 0;
	size = 0;
	access_time = 0;
	mod_time = 0;
	attrib_time = 0;

	uid = 0;
	uid_name = "";
	gid = 0;
	gid_name = "";
}



//-----------------------------------------------------------------------------
file_info& file_info::operator= (string& rhs)
{
	name = rhs;

	init();

	read_file_info();

	return *this;
}



//-----------------------------------------------------------------------------
file_info& file_info::operator= (const char* rhs)
{
	name = rhs;

	init();

	read_file_info();

	return *this;
}




//-----------------------------------------------------------------------------
file_info::file_info(const string& src)
{
	name = src;

	init();

	read_file_info();
}



//-----------------------------------------------------------------------------
file_info::file_info(const char* src)
{
	name = src;

	init();

	read_file_info();
}



//-----------------------------------------------------------------------------
file_info::file_info()
{
	name = "";

	init();
}
