/* $Id: ipc-daemon.c,v 1.21 2003/07/13 20:27:50 cwilson Exp $ */

/*
 *   IPC package for CygWin
 *
 *   Copyright (C) 1997 Philippe CHAPUY
 *   Copyright (C) 1998 Ludovic LANGE
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 *   HISTORY:
 *   --------
 *
 *   13/05/1998 : Version 1.00 released
 *                First public release
 *                adress any comments to llange@capgemini.fr
 *
 */

/*
 * About fprintf, syslog, and log_message:
 *  if a message needs to be sent to the user, then use these
 *  functions under the following circumstances:
 *  - fprintf: stuff that is immediately user-interactive
 *      e.g. you don't want to see "help instructions" 
 *      in the syslog.
 *  - syslog: when the block of code is only reachable
 *      when running as a service --- but, for simplicity,
 *      we just use log_message anyway.
 *  - log_message: when the block of code could be reached
 *      when running as a service OR standalone
 */

/************************************************************************/
/*									*/
/************************************************************************/

#define EXTERN
#include <IpcNtExt.h>
#include <unistd.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <stdarg.h>
#include <math.h>
#include <sys/ipctrace.h>
#include <exceptions.h>
#include <sys/cygwin.h>
#include <sys/wait.h>
#include <malloc.h>
#include <syslog.h>
#include <popt.h>

static void printTopDescription(FILE * f, char * name);
static void printBottomDescription(FILE * f, char * name);
static char * getVersion(char * buf, int len);
static void usage(poptContext optCon, FILE * f, char * name);
static void help(poptContext optCon, FILE * f, char * name);
static void version(poptContext optCon, FILE * f, char * name);
static void license(poptContext optCon, FILE * f, char * name);
int log_message(FILE *stream, int severity, const char *fmt, ...);
int libcygipc_err_handler(FILE *stream, int severity, const char *fmt, ...);
void get_null_sa (SECURITY_ATTRIBUTES *sap);
int get_reg_entries(void);
int install_reg_entries(const char *myname, int verbose, int store_verbose);

char	verbose=1;
char	service=0;
char	tight_security=0;
static char * program_name;

#define is_winnt	(GetVersion() < 0x80000000)

# define IPCD_SERVICE_NAME	"ipc-daemon2"
# define IPCD_SERVICE_DISPLAY	"Cygwin IPC Daemon 2"

#define IPCD_PARAM_KEY	"SYSTEM\\CurrentControlSet\\Services\\" \
                        IPCD_SERVICE_NAME \
                        "\\Parameters"
#define IPCD_PARAM_SECURITY	"TightSecurity"
#define IPCD_PARAM_VERBOSE		"Verbose"

int server_pid = 0;
SERVICE_STATUS_HANDLE ssh;
SERVICE_STATUS ss;
HANDLE hStopEvent = NULL;
/* handles to semaphores, so they stay in existence 
 * as long as daemon is running
 */
HANDLE *semaphore_handles[SEMMNI];

static void msg_init (CYGWIN_IPCNT_MSGSTR *ShareAdr)
{
	int id, msg;

	for (id = 0; id < MSGMNI; id++)
	{
	 ShareAdr->msgque[id] = (struct msqid_ds *) IPC_UNUSED;
	 for (msg = 0; msg < MSGMAXMESS; msg++)
	 {
	  ShareAdr->msg[id].msg[msg].msg_spot =
	  	&(ShareAdr->msg[id].msg[msg].data[0]) - (int) ShareAdr ;
	 }
	}
	ShareAdr->msgbytes = ShareAdr->msghdrs = ShareAdr->msg_seq =
	ShareAdr->used_queues = 0;

	return;
}

static void shm_init (CYGWIN_IPCNT_SHMSTR *ShareAdr)
{
	int id, i;

	for (id = 0; id < SHMMNI; id++)
	{
	 ShareAdr->shm_segs[id] = (struct shmid_ds *) IPC_UNUSED;
	 for (i = 0; i < SHMMNI; i++)
	 {
	  ShareAdr->shm[id].attaches[i].pid = 0 ;
	  ShareAdr->shm[id].attaches[i].fd  = 0 ;
	  ShareAdr->shm[id].attaches[i].adr = 0 ;
	 }
	}
	ShareAdr->shm_rss = ShareAdr->shm_seq =
	ShareAdr->max_shmid = 0;
	return;
}

static void sem_init (CYGWIN_IPCNT_SEMSTR *ShareAdr)
{
	int i;

	ShareAdr->used_sems = ShareAdr->used_semids =
	ShareAdr->max_semid = ShareAdr->sem_seq = 0;
	for (i = 0; i < SEMMNI; i++)
	{
		ShareAdr->state[i] = 0 ;
		ShareAdr->semary[i] = (struct semid_ds *) IPC_UNUSED;
	}
	return;
}

static void destroy_map(const char *pcName,
	int *pnFd, caddr_t *ppMap, size_t nSize,
	const char *pcFile)
{
	if (ppMap != NULL && *ppMap != (caddr_t)-1) {
		if (munmap(*ppMap, nSize) < 0) {
			log_message(stderr, LOG_ERR,
				"Unable to unmap \"%s\" file: %s",
				pcName, strerror(errno));
		}
		*ppMap = (caddr_t)-1;
	}

	if (pnFd != NULL && *pnFd >= 0) {
		if (close(*pnFd) < 0) {
			log_message(stderr, LOG_ERR,
				"Unable to close \"%s\" file: %s",
				pcName, strerror(errno));
		}
		*pnFd = -1;

		if (pcFile != NULL) {
			if (unlink(pcFile) < 0) {
				log_message(stderr, LOG_ERR,
					"Unable to remove \"%s\" file: %s",
					pcName, strerror(errno));
			}
		}
	}

	return;
}

static int create_map(const char *pcName,
	int *pnFdRet, caddr_t *ppMapRet, size_t nSize,
	const char *pcFile)
{
	int		iRv;
	int		nFd = -1;
	void	*pvBuf = NULL;
	caddr_t	pMap = (caddr_t)-1;

	do {	/* dummy loop */
		iRv = -1;

		nFd = open(pcFile, O_CREAT|O_RDWR, 0666 );  /* O_TRUNC ? */
		if (nFd < 0) {
			log_message(stderr, LOG_ERR,
				"Unable to open \"%s\" file: %s",
				pcName, strerror(errno));
			break;
		}

		pvBuf = malloc(nSize);
		if (pvBuf == NULL) {
			log_message(stderr, LOG_ERR,
				"Unable to allocate \"%s\" memory: %s",
				pcName, strerror(errno));
			break;
		}

		memset(pvBuf, 0, nSize) ;
		if (write(nFd, pvBuf, nSize) != (ssize_t)nSize) {
			log_message(stderr, LOG_ERR,
				"Unable to write \"%s\" file: %s",
				pcName, strerror(errno));
			break;
		}
		close(nFd);

		nFd  = open(pcFile, O_RDWR, 0666);
		if (nFd < 0) {
			log_message(stderr, LOG_ERR,
				"Unable to open \"%s\" file again: %s",
				pcName, strerror(errno));
			break;
		}
		pMap = mmap(0, nSize, PROT_READ|PROT_WRITE, MAP_SHARED, nFd, 0);
		if (pMap == (caddr_t)-1) {
			log_message(stderr, LOG_ERR,
				"Unable to mmap \"%s\" file: %s",
				pcName, strerror(errno));
			break;
		}

		iRv = 0;
	} while(0);	/* end dummy loop */
	
	if (pvBuf != NULL) {
		free(pvBuf);
	}

	if (iRv < 0) {
		if (nFd >= 0) {
			close(nFd);
			nFd = -1;
		}
	}

	*pnFdRet = nFd;
	*ppMapRet = pMap;

	return iRv;
}

int work_main(void)
{
	int iRv;
	HANDLE LSemControl = NULL;
	int LFdSem = -1, LFdShm = -1, LFdMsg = -1;
	CYGWIN_IPCNT_SEMSTR *LAdrSem = (CYGWIN_IPCNT_SEMSTR *)-1;
	CYGWIN_IPCNT_SHMSTR *LAdrShm = (CYGWIN_IPCNT_SHMSTR *)-1;
	CYGWIN_IPCNT_MSGSTR *LAdrMsg = (CYGWIN_IPCNT_MSGSTR *)-1;
	SECURITY_ATTRIBUTES sa;

	HANDLE LHandle ;
	char LBuff[100] ;
	int id, Index ;
	struct semid_ds *sma ;
	long LPrevious ;
	int LComptShm = 0 ;
	int LRet ;

	do {	/* dummy loop */
		iRv = 1;

		get_null_sa(&sa);

		LSemControl = OpenSemaphore(SEMAPHORE_ALL_ACCESS, FALSE,
				CYGWIN_IPCNT_SEMCTL) ;
		if (LSemControl != NULL) {
			log_message(stderr, LOG_ERR, "IPC-daemon is already started !!") ;
			iRv = 2;
			break;
		}

		LSemControl = CreateSemaphore(tight_security ? NULL : &sa, 0, 1,
				CYGWIN_IPCNT_SEMCTL) ;
		if (LSemControl == NULL) {
			log_message(stderr, LOG_ERR, "Unable to create a semaphore");
			break;
		}

		/*********************************************************************/
		/* Creation des semaphore de gestion des memoires partagee           */
		/* Si il existe dans le systeme, On les vire                         */
		/*********************************************************************/
		GSemSem = OpenSemaphore(SEMAPHORE_ALL_ACCESS, FALSE,
				CYGWIN_IPCNT_SEMSEM) ;
		if (GSemSem != NULL) {
			CloseHandle(GSemSem);
		}

		GSemSem = CreateSemaphore(tight_security ? NULL : &sa, 1, 1,
				CYGWIN_IPCNT_SEMSEM) ;

		if (GSemSem == NULL) {
			log_message(stderr, LOG_ERR, "Unable to create \"Sem\" semaphore");
			break;
		}

		GSemShm = OpenSemaphore(SEMAPHORE_ALL_ACCESS, FALSE,
				CYGWIN_IPCNT_SEMSHM);
		if (GSemShm != NULL) {
			CloseHandle(GSemShm);
		}

		GSemShm = CreateSemaphore(tight_security ? NULL : &sa, 1, 1,
				CYGWIN_IPCNT_SEMSHM) ;
		if (GSemShm == NULL) {
			log_message(stderr, LOG_ERR, "Unable to create \"Shm\" semaphore");
			break;
		}

		GSemMsg = OpenSemaphore(SEMAPHORE_ALL_ACCESS, FALSE,
				CYGWIN_IPCNT_SEMMSG);
		if (GSemMsg != NULL) {
			CloseHandle(GSemMsg);
		}

		GSemMsg = CreateSemaphore(tight_security ? NULL : &sa, 1, 1,
				CYGWIN_IPCNT_SEMMSG) ; ;
		if (GSemMsg == NULL) {
			log_message(stderr, LOG_ERR, "Unable to create \"Msg\" semaphore");
			break;
		}

		/*********************************************************************/
		/* Creation des memoires partagees                                   */
		/*********************************************************************/

		/*********************************************************************/
		/* On commence par le semaphore                                      */
		/*********************************************************************/
		if (create_map("Sem", &LFdSem, (caddr_t*)&LAdrSem
				, sizeof(CYGWIN_IPCNT_SEMSTR), CYGWIN_IPCNT_FILESEM
		) < 0) {
			break;
		}
		sem_init(LAdrSem) ;

		/*********************************************************************/
		/* On continue par la memoire partagee                               */
		/*********************************************************************/
		if (create_map("Shm", &LFdShm, (caddr_t*)&LAdrShm
				, sizeof(CYGWIN_IPCNT_SHMSTR), CYGWIN_IPCNT_FILESHM
		) < 0) {
			break;
		}
		shm_init(LAdrShm) ;

		/*********************************************************************/
		/* On finit par les messages queues                                  */
		/*********************************************************************/
		if (create_map("Msg", &LFdMsg, (caddr_t*)&LAdrMsg
				, sizeof(CYGWIN_IPCNT_MSGSTR), CYGWIN_IPCNT_FILEMSG
		) < 0) {
			break;
		}
		msg_init(LAdrMsg) ;

		hStopEvent = CreateEvent(tight_security ? NULL : &sa,
				TRUE, FALSE, NULL);
		if (hStopEvent == NULL) {
			log_message(stderr, LOG_ERR, "failed to create stop event");
			break;
		}

		if (service != 0) {
			ss.dwCheckPoint = 0;
			ss.dwWaitHint = 0;
			ss.dwCurrentState = SERVICE_RUNNING;
			SetServiceStatus(ssh, &ss);
			log_message(stderr, LOG_INFO, 
				"'" IPCD_SERVICE_NAME "' service started"); 
		}

		iRv = 0;
	} while(0);	/* end dummy loop */

	/*********************************************************************/
	/* On boucle afin de recuperer les Handle des objets crees           */
	/* par les fonctions IPC                                             */
	/*********************************************************************/

	/* do not enter if not initialized successfully */
	while(iRv == 0) {
		HANDLE hObjects[2];
		DWORD ret;
		usleep(300000) ;

		/********************************************************************/
		/* Semaphores                                                       */
		/********************************************************************/

		hObjects[0] = hStopEvent;
		hObjects[1] = GSemSem;
		/* The order of the objects matters, since we want the stop-event
		-* to take precedence over the GSemSem-semaphore which is almost
		-* always signalled.
		*/

		ret = WaitForMultipleObjects(2, hObjects, FALSE, INFINITE) ;

		/* switch(ret) */
		/* case WAIT_OBJECT_0: */
		if (ret == WAIT_OBJECT_0) {
			/* got the stop-event */
			break;
		}

		/* default: */
		if (ret != WAIT_OBJECT_0+1) {
			log_message(stderr, LOG_ERR,
				"unexpected return from wait, %ld", ret);
			iRv = 1;
			break;
		}

		/* case WAIT_OBJECT_0+1: */
		for (id = 0; id < SEMMNI; id++) {
			if (LAdrSem->semary[id] == IPC_UNUSED) {
				continue;
			}
			if ( (LAdrSem->state[id] != 1) && (LAdrSem->state[id] != 0) ){
				continue;
			}
			sma = (struct semid_ds *) (
				(char *) LAdrSem->semary[id] + (int) LAdrSem
			);
			if (LAdrSem->state[id] == 0) { /* ready to be used */
				semaphore_handles[id] = malloc(sma->sem_nsems * sizeof(HANDLE));
				for (Index = 0; Index < sma->sem_nsems; Index++) {
					name_mangle(100*id+Index, LBuff) ;
					semaphore_handles[id][Index] = OpenSemaphore(
						SEMAPHORE_ALL_ACCESS, FALSE, LBuff) ; 
				}
				LAdrSem->state[id] = 2 ; /* normal operation, handles alloced */
				continue;
			}

			/* destroyed: lock semaphores if it still exists */
			for (Index = 0; Index < sma->sem_nsems; Index++) {
				name_mangle(100*id+Index, LBuff);
				LHandle = OpenSemaphore(SEMAPHORE_ALL_ACCESS, FALSE, LBuff);
				if (LHandle != NULL) {
					/* someone has this semaphore still open, so lock it
					*/
					while (WaitForSingleObject(LHandle, 0) == WAIT_OBJECT_0)
						;
					CloseHandle(LHandle) ;
				}
				if (semaphore_handles[id][Index] != NULL) {
					/* someone has this semaphore still open, so lock it
					*/
					while (WaitForSingleObject(semaphore_handles[id][Index], 0) == 
					       WAIT_OBJECT_0)
						;
					CloseHandle(semaphore_handles[id][Index]) ;
				}
				LAdrSem->current_nb[id].current_nb[Index] = 0;
			}
			free(semaphore_handles[id]);
			semaphore_handles[id] = 0;
			LAdrSem->semary[id] = (struct semid_ds *) IPC_UNUSED;
			LAdrSem->state[id]  = 0; /* ready to be used */
		}

		ReleaseSemaphore(GSemSem, (LONG) 1, &LPrevious) ;

		/********************************************************************/
		/* Memoires partagees                                               */
		/* On vire les memoires partagees non referencees                   */
		/* de temps en temps, on vire les shm dont les pid n'existent plus  */
		/********************************************************************/

		if (LComptShm < 10) {
			LComptShm++ ;
			continue;
		}
		LComptShm = 0;
		WaitForSingleObject(GSemShm, INFINITE);

		for (id = 0; id < SHMMNI; id++) {
			int nAttached = 0;

			if( LAdrShm->shm_segs[id] == (struct shmid_ds *)IPC_UNUSED) {
				continue;
			}

			for (Index=0; Index<SHMMNI; Index++) {
				if (LAdrShm->shm[id].attaches[Index].pid == 0) {
					continue;
				}

				LRet = kill(LAdrShm->shm[id].attaches[Index].pid, 0);
				if (LRet == 0) {
					/* attached process is still alive */
					++ nAttached;
					continue;
				}
				/* attached process died without shmdt.
				-* do NOT unmap() shm for detached process,
				-* and do NOT close() file for detached process.
				-* just invalidate the attaches-entry.
				*/
				// munmap(LAdrShm->shm[id].attaches[Index].adr , 
				//        LAdrShm->shm[id].shm_segsz ) ;
				// close (LAdrShm->shm[id].attaches[Index].fd) ;
				LAdrShm->shm[id].attaches[Index].pid = 0 ;
				LAdrShm->shm[id].attaches[Index].adr = 0 ;
				LAdrShm->shm[id].attaches[Index].fd  = 0 ;
				LAdrShm->shm[id].shm_nattch--;      /* for destruction */
			}
			/* no attached processes any more,
			-* but shmctl(IPC_RMID) was called while
			-* other processes did not shmdt() yet,
			-* destroy it now.
			*/
			if (LAdrShm->shm[id].shm_nattch <= 0
			 && (LAdrShm->shm[id].shm_perm.mode & SHM_DEST) == SHM_DEST
			) {
				char LBuff[100] ;

				LAdrShm->shm[id].shm_perm.seq++;
				LAdrShm->shm_seq =
					(LAdrShm->shm_seq+1) % ((unsigned)(1<<31)/SHMMNI);
				LAdrShm->shm_segs[id] = (struct shmid_ds *) IPC_UNUSED;

				strcpy(LBuff, "/tmp/") ;
				name_mangle(id, LBuff+strlen(LBuff)) ;
				unlink(LBuff);
			}
		}

		ReleaseSemaphore(GSemShm, (LONG) 1, &LPrevious) ;
	}

	if (iRv != 0) {
		log_message(stderr, LOG_ERR, "Unable to start IPC-daemon !") ;
	}

	destroy_map("Msg", &LFdMsg, (caddr_t*)&LAdrMsg
			, sizeof(CYGWIN_IPCNT_MSGSTR), CYGWIN_IPCNT_FILEMSG);
	destroy_map("Shm", &LFdShm, (caddr_t*)&LAdrShm
			, sizeof(CYGWIN_IPCNT_SHMSTR), CYGWIN_IPCNT_FILESHM);
	destroy_map("Sem", &LFdSem, (caddr_t*)&LAdrSem
			, sizeof(CYGWIN_IPCNT_SEMSTR), CYGWIN_IPCNT_FILESEM);

	if (LSemControl != NULL) {
		CloseHandle(LSemControl);
	}

	return iRv; 
}
 
int install_reg_entries(const char *myname, int verbose, int store_verbose)
{
  HKEY srv_key = NULL;
  DWORD disp;
  DWORD temp;
  char *err_func;

  if (RegCreateKeyEx(HKEY_LOCAL_MACHINE, IPCD_PARAM_KEY, 0, "",
                     REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS,
                     NULL, &srv_key, &disp)
      != ERROR_SUCCESS)
    {
      err_func = "RegCreateKeyEx";
      goto error;
    }
  temp = tight_security;
  if (RegSetValueEx(srv_key, IPCD_PARAM_SECURITY, 0, REG_DWORD,
                    (CONST BYTE *) &temp, sizeof(DWORD))
      != ERROR_SUCCESS)
    {
      err_func = "RegSetValueEx(TightSecurity=1)";
      goto error;
    }
  temp = store_verbose;
  if (RegSetValueEx(srv_key, IPCD_PARAM_VERBOSE, 0, REG_DWORD,
                    (CONST BYTE *) &temp, sizeof(DWORD))
      != ERROR_SUCCESS)
    {
      err_func = "RegSetValueEx(Verbose=x)";
      goto error;
    }
  RegFlushKey(srv_key);
  RegCloseKey(srv_key);
  return 0;
error:
  if (srv_key)
    RegCloseKey(srv_key);
  if (verbose)
    log_message(stderr, LOG_ERR, "%s: %ld = %s()\n",
            myname, GetLastError(), err_func);
  log_message(stderr, LOG_ERR, 
     "%s: installing registry entries for `inetd' failed\n",
     myname);
  return 1;
}

int get_reg_entries(void)
{
  HKEY srv_key = NULL;
  DWORD type, size;
  DWORD tight_security_temp;
  DWORD verbose_temp;

  if (RegOpenKeyEx(HKEY_LOCAL_MACHINE, IPCD_PARAM_KEY, 0,
                   KEY_ALL_ACCESS, &srv_key)
      != ERROR_SUCCESS)
    {
      log_message(stderr, LOG_WARNING, 
		   "couldn't get parameters from registry, using defaults\n"
         "error %ld in RegOpenKeyEx()", GetLastError());
      return 1;
    }
  if (RegQueryValueEx(srv_key, IPCD_PARAM_VERBOSE, 0, &type,
                      (unsigned char *) &verbose_temp, (size = sizeof(DWORD), &size))
      != ERROR_SUCCESS)
    log_message(stderr, LOG_WARNING, 
	    "couldn't get verbose value from registry, "
      "using default value %d", verbose);
  else {
    verbose = (char) verbose_temp;
	 log_message(stderr, LOG_DEBUG, "verbose value: %d", verbose);
  }
  if (RegQueryValueEx(srv_key, IPCD_PARAM_SECURITY, 0, &type,
      (unsigned char *) &tight_security_temp, (size = sizeof(DWORD), &size))
      != ERROR_SUCCESS)
    log_message(stderr, LOG_WARNING, 
	    "couldn't get tight_security boolean from registry, "
       "using default value %d", tight_security);
  else {
    tight_security = (char) tight_security_temp;
    log_message(stderr, LOG_DEBUG, "tight_security value: %d", tight_security);
  }
  RegCloseKey(srv_key);
  return 0;
}

VOID WINAPI
service_handler(DWORD ctrl)
{
       exception_list except_list;
       cygwin_internal (CW_INIT_EXCEPTIONS, &except_list);
       switch (ctrl) {
       case SERVICE_CONTROL_STOP:
               ss.dwCheckPoint = 1;
               ss.dwWaitHint = 3000L; // wait up to 3 seconds...
               ss.dwCurrentState = SERVICE_STOP_PENDING;

               if (hStopEvent) {
                 log_message(stderr, LOG_INFO, "sending hStopEvent");
                 SetEvent(hStopEvent);
               }
               else
                 log_message(stderr, LOG_INFO, "no hStopEvent to send");

               break;
       default:
               // ignore
               break;
       }
       SetServiceStatus(ssh, &ss);
}

void WINAPI
service_main(DWORD argc, LPSTR *argv)
{
        exception_list except_list;
        cygwin_internal (CW_INIT_EXCEPTIONS, &except_list);
       // Initialized Cygwin exception handling for thread not
       // created by Cygwin.

       if (! (ssh = RegisterServiceCtrlHandlerA(IPCD_SERVICE_NAME,
                                 service_handler))) {
                log_message(stderr, LOG_ERR,
                       "error in service_main: %d = RegSrvCtrlHandler()",
                       GetLastError());
                log_message(stderr, LOG_ERR, "starting '" IPCD_SERVICE_NAME "' as a service failed");
                return;
       }
       ss.dwServiceType = SERVICE_WIN32_SHARE_PROCESS;
       ss.dwCurrentState = SERVICE_START_PENDING;
       ss.dwControlsAccepted = SERVICE_ACCEPT_STOP;
       ss.dwWin32ExitCode = NO_ERROR;
       ss.dwCheckPoint = 1;
       ss.dwWaitHint = 3000L;
       SetServiceStatus(ssh, &ss);

       get_reg_entries();
		 log_message(stderr, LOG_DEBUG, "service: %d, verbose: %d, tight_security: %d",
		   service, verbose, tight_security);
       work_main();

       log_message(stderr, LOG_INFO, "'" IPCD_SERVICE_NAME "' service stopped");
       ss.dwCheckPoint = 0;
       ss.dwCurrentState = SERVICE_STOPPED;
       SetServiceStatus(ssh, &ss);
}

int
install_as_service(const char *myname, int verbose)
{
  char mypath[MAX_PATH+5];
  char mycmd[MAX_PATH+15];
  char *err_func;
  SC_HANDLE sm = (SC_HANDLE) 0;
  SC_HANDLE sh = (SC_HANDLE) 0;
  SC_LOCK sl = (SC_LOCK) 0;

  // get own full path name
  if (! GetModuleFileName(NULL, mypath, MAX_PATH))
    {
      err_func = "GetModuleFileName";
      goto error;
    }
  // we can't read parameters from registry unless we
  // already know we are a service.  So, appname must include
  // --service option:
  mycmd[0]='"';
  mycmd[1]='\0';
  strcat(mycmd, mypath);
  strcat(mycmd, "\" \"--service\"");
  // open service manager database
  if (! (sm = OpenSCManager(NULL, NULL, SC_MANAGER_ALL_ACCESS)))
    {
      err_func = "OpenSCManager";
      goto error;
    }
  // lock service database
  if (! (sl = LockServiceDatabase(sm)))
    {
      err_func = "LockServiceDatabase";
      goto error;
    }
  // open service, go ahead if service doesn't exists
  if (! (sh = OpenService(sm, IPCD_SERVICE_NAME, SERVICE_ALL_ACCESS)) &&
      GetLastError() != ERROR_SERVICE_DOES_NOT_EXIST)
    {
      err_func = "OpenService";
      goto error;
    }
  // service already exists
  if (sh)
    {
      if (verbose)
        log_message(stderr, LOG_WARNING, "%s: service already installed", myname);
    }
  // no, create service
  else if (! (sh = CreateService(sm,
                                 IPCD_SERVICE_NAME,
                                 IPCD_SERVICE_DISPLAY,
                                 STANDARD_RIGHTS_REQUIRED |
                                 SERVICE_INTERROGATE |
                                 SERVICE_QUERY_STATUS |
                                 SERVICE_START |
                                 SERVICE_STOP,
                                 SERVICE_WIN32_OWN_PROCESS /*|
                                 SERVICE_INTERACTIVE_PROCESS*/,
                                 SERVICE_AUTO_START,
                                 SERVICE_ERROR_NORMAL,
                                 mycmd,
                                 NULL,
                                 NULL,
                                 NULL,
                                 NULL,
                                 NULL
                                )))
    {
      err_func = "CreateService";
      goto error;
    }

  // if CloseServiceHandle returns with error, it's not sure
  // whether anything works...
  if (! CloseServiceHandle(sh))
    {
      err_func = "CloseServiceHandle";
      goto error;
    }
  return 0;
error:
  if (verbose)
    log_message(stderr, LOG_ERR, "%s: %lu = %s()",
            myname, GetLastError(), err_func);
  log_message(stderr, LOG_ERR, "%s: installing service `" IPCD_SERVICE_NAME "' failed", myname);
  if (sh)
    CloseServiceHandle(sh);
  if (sl && ! UnlockServiceDatabase(sl))
    log_message(stderr, LOG_WARNING,
            "%s: CAUTION: UnlockServiceDatabase() failed "
            "with error %lu", myname, GetLastError());
  return 1;
}

int
remove_as_service(const char *myname, int verbose)
{
  char *err_func;
  SC_HANDLE sm = (SC_HANDLE) 0;
  SC_HANDLE sh = (SC_HANDLE) 0;
  SC_LOCK sl = (SC_LOCK) 0;
  SERVICE_STATUS ss;
  int err;

  // open service manager database
  if (! (sm = OpenSCManager(NULL, NULL, SC_MANAGER_ALL_ACCESS)))
    {
      err_func = "OpenSCManager";
      goto error;
    }
  // lock service database
  if (! (sl = LockServiceDatabase(sm)))
    {
      err_func = "LockServiceDatabase";
      goto error;
    }
  // check whether service exists
  if (! (sh = OpenService(sm, IPCD_SERVICE_NAME, SERVICE_ALL_ACCESS)))
    {
      if (GetLastError() == ERROR_SERVICE_DOES_NOT_EXIST)
        {
          if (verbose)
            log_message(stderr, LOG_ERR, "%s: service isn't installed", myname);
          if (! UnlockServiceDatabase(sl))
            log_message(stderr, LOG_WARNING,
                    "%s: CAUTION: UnlockServiceDatabase() failed "
                    "with error %lu", myname, GetLastError());
            return 0;
        }
      err_func = "OpenService";
      goto error;
    }
  // try stopping service
  // if NOT_ACTIVE or CANNOT_ACCEPT_CTRL go ahead, else exit
  if (! ControlService(sh, SERVICE_CONTROL_STOP, &ss) &&
      (err = GetLastError()) != ERROR_SERVICE_NOT_ACTIVE &&
       err != ERROR_SERVICE_CANNOT_ACCEPT_CTRL)
    {
      err_func = "ControlService";
      goto error;
    }
  // wait, 'til service is stopped
  do
    {
      sleep(1);
      if (! QueryServiceStatus(sh, &ss))
        {
          err_func = "QueryServiceStatus";
          goto error;
        }
    }
  while (ss.dwCurrentState != SERVICE_STOPPED);
  // remove service
  if (! DeleteService(sh))
    {
      err_func = "DeleteService";
      goto error;
    }
  // if CloseServiceHandle returns with error, it's not sure
  // whether anything works...
  if (! CloseServiceHandle(sh))
    {
      sh = (SC_HANDLE) 0;
      err_func = "CloseServiceHandle";
      goto error;
    }
  sh = (SC_HANDLE) 0;
  // Datenbank-Lock aufheben
  if (sl && ! UnlockServiceDatabase(sl))
    {
      log_message(stderr, LOG_WARNING,
              "%s: CAUTION: UnlockServiceDatabase() failed "
              "with error %lu", myname, GetLastError());
    }
  return 0;
error:
  if (verbose)
    log_message(stderr, LOG_ERR, "%s: %lu = %s()",
            myname, GetLastError(), err_func);
  log_message(stderr, LOG_ERR, "%s: removing `" IPCD_SERVICE_NAME "' as service failed", myname);
  if (sh)
    CloseServiceHandle(sh);
  if (sl && ! UnlockServiceDatabase(sl))
    fprintf(stderr,
            "%s: CAUTION: UnlockServiceDatabase() failed "
            "with error %lu", myname, GetLastError());
  return 1;
}

int main(int argc, const char ** argv)
{
  int install = 0, remove = 0;
  poptContext optCon;
  const char ** rest;
  int rc;
  int ec = 0;

  struct poptOption generalOptionsTable[] = {
    { "install-as-service", '\0', POPT_ARG_NONE, NULL, 'I', \
        "Install as a windows service.", NULL},
    { "remove-as-service", '\0', POPT_ARG_NONE, NULL, 'R', \
        "Remove as a windows service.", NULL},
    { "service", 's', POPT_ARG_NONE, NULL, 's', \
        "Run in service mode.  This option should ONLY be used "
        "from within the NT/2K service manager system; do NOT "
        "use it in ANY other context -- shortcuts, .bat files, "
        "manual command-line entry, etc.", NULL},
    { "tight-security", 't', POPT_ARG_NONE, NULL, 't', \
        "Run in restrictive security mode.", NULL},
    { "debug-level", 'D', POPT_ARG_INT, &verbose, 'D', \
      "Set verboseness of output.  0=quiet mode (also --quiet or -q), while "
      "2=print execution trace to stderr (or syslog if in service mode) "
		"(also --debug or -d)",
      "<int>"},
    POPT_TABLEEND
  };

  struct poptOption helpOptionsTable[] = {
    { "help",  '?',  POPT_ARG_NONE, NULL, '?', \
        "Show this help message", NULL},
    { "usage", '\0', POPT_ARG_NONE, NULL, 'u', \
        "Display brief usage message", NULL},
    { "version", '\0', POPT_ARG_NONE, NULL, 'v', \
        "Display version information", NULL},
    { "license", '\0', POPT_ARG_NONE, NULL, 'l', \
        "Display licensing information", NULL},
    POPT_TABLEEND
  };

  struct poptOption opt[] = {
    { NULL, '\0', POPT_ARG_INCLUDE_TABLE, generalOptionsTable, 0, \
        "General options", NULL },
    { NULL, '\0', POPT_ARG_INCLUDE_TABLE, helpOptionsTable, 0, \
        "Help options", NULL },
    POPT_TABLEEND
  };

  struct poptAlias debugAlias = {
    "debug", 'd', 0, NULL
  };
  struct poptAlias quietAlias = {
    "quiet", 'q', 0, NULL
  };

  openlog(IPCD_SERVICE_NAME, LOG_PID | LOG_NOWAIT, LOG_DAEMON);
  cygipc_set_err_handler(libcygipc_err_handler);

  if( (program_name = strdup(argv[0])) == NULL ) {
    log_message(stderr, LOG_ERR, "%s: memory allocation error", argv[0]);
    exit(1);
  }

  if (poptParseArgvString("--debug-level 2",&(debugAlias.argc),&(debugAlias.argv))) {
    log_message(stderr, LOG_ERR, "%s: Problem creating debug alias", argv[0]);
    exit(1);
  }
  if (poptParseArgvString("--debug-level 0",&(quietAlias.argc),&(quietAlias.argv))) {
    log_message(stderr, LOG_ERR, "%s: Problem creating quiet alias", argv[0]);
    exit(1);
  }
  optCon = poptGetContext(IPCD_SERVICE_NAME, argc, argv, opt, 0);

  if( poptAddAlias(optCon, debugAlias, 0) ) {
    log_message(stderr, LOG_ERR, "%s: Problem adding debug alias", argv[0]);
    exit(1);
  }
  if( poptAddAlias(optCon, quietAlias, 0) ) {
    log_message(stderr, LOG_ERR, "%s: Problem adding quiet alias", argv[0]);
    exit(1);
  }
  while ((rc = poptGetNextOpt(optCon)) > 0) {
    switch (rc) {
      case '?':  help(optCon, stderr, program_name);
                 goto exit;
      case 'u':  usage(optCon, stderr, program_name);
                 goto exit;
      case 'v':  version(optCon, stderr, program_name);
                 goto exit;
      case 'l':  license(optCon, stderr, program_name);
                 goto exit;
      case 'I':  install = 1;
                 break;
      case 'R':  remove = 1;
                 break;
      case 's':  service = 1;
                 break;
      case 't':  tight_security = 1;
                 break;
      case 'D':  if (verbose < 0) { verbose = 0; }
                 if (verbose > 2) { verbose = 2; }
                 if (verbose >= 2) { cygipc_set_debug(1); }
                 if (verbose <= 0) { cygipc_set_debug(0); }
                 break;
    }
  }
  if (rc < -1 ) {
    log_message(stderr, LOG_ERR, "%s: bad argument %s: %s",
      program_name, poptBadOption(optCon, POPT_BADOPTION_NOALIAS),
      poptStrerror(rc));
    ec = 2;
    goto exit;
  }
  if (install && remove) {
    log_message(stderr, LOG_ERR, 
      "%s: can't specify both --install-as-service and --remove-as-service\nNo action taken.", 
      program_name);
    ec = 2;
    goto exit;
  }

  rest = poptGetArgs(optCon);
  if (rest) {
    char buf[2000];
    int count;
    count = snprintf(buf, 2000, "Extra args ignored: ");
    while (*rest) 
      count += snprintf(buf + count, 2000 - count, "%s ", *rest++);
    log_message(stderr, LOG_WARNING, buf);
  }

  if (install) {
    ec = install_as_service(program_name, 1);
    if (!ec) {
      ec = install_reg_entries(program_name, 1, verbose);
      if (ec)
        remove_as_service(program_name, 1);
    }
  }
  else if (remove) {
    ec = remove_as_service(program_name, 1);
  }
  else if (service) {
    SERVICE_TABLE_ENTRYA ste[2];
    ste[0].lpServiceName = IPCD_SERVICE_NAME;
    ste[0].lpServiceProc = service_main;
    ste[1].lpServiceName = NULL;
    ste[1].lpServiceProc = NULL;
    StartServiceCtrlDispatcherA(ste);
    ec = 0;
  }
  else {
    work_main();
    ec = 0;
  }

exit:
  poptFreeContext(optCon);
  free(program_name);
  closelog();
  return(ec);
}

int log_message(FILE *stream, int severity, const char *fmt, ...)
{
  char buf[2000];
  va_list ap;
  int count;

  if( (verbose >= 2) || (severity < LOG_DEBUG))
  {
    count = snprintf (buf, 2000, "(" IPCD_SERVICE_NAME ") ");
    va_start (ap, fmt);
    count += vsnprintf (buf + count, 2000 - count, fmt, ap);
    va_end (ap);
    if(service)
      syslog(severity, buf);
    else
    {
      snprintf(buf + count, 2000 - count, "\n");
      fprintf(stream, buf);
    }
  }
  return 0;
}

int libcygipc_err_handler(FILE *stream, int severity, const char *fmt, ...)
{
  char buf[2000];
  va_list ap;
  int count;

  if(cygipc_get_debug() || (severity < LOG_DEBUG))
  {
    count = snprintf (buf, 2000, "(" IPCD_SERVICE_NAME ":libcygipc) ");
    va_start (ap, fmt);
    count += vsnprintf (buf + count, 2000 - count, fmt, ap);
    va_end (ap);
    if(service)
      syslog(severity, fmt);
    else
    {
      snprintf(buf + count, 2000 - count, "\n");
      fprintf(stream, buf);
    }
  }
  return 0;
}

static char * getVersion(char * buf, int len)
{
  unsigned short LVersionMajor=1;
  unsigned short LVersionMinor=0;

  LVersionMajor=(unsigned short) floor(VERSION_NUM);
  LVersionMinor=(unsigned short) rint( (100.0 * VERSION_NUM) - (100.0*((double)LVersionMajor)) );
  snprintf(buf, len, "%d.%02d", LVersionMajor, LVersionMinor);
  return buf;
}

static void printTopDescription(FILE * f, char * name)
{
  char buf[512];
  fprintf(f, "%s : Cygwin IPC Package Version %s (c) 1998 PCH/LLA, Changes (c) 2001 CSW\n", 
          name, getVersion(buf, 512));
  fprintf(f, "  This daemon provides InterProcess Communication services for Cygwin.\n\n");
}
static void printBottomDescription(FILE * f, char * name)
{
  fprintf(f,"\n");
  fprintf(f, "Other acceptable option aliases:\n");
  fprintf(f,"  -d, --debug                 sets --debug-level=2\n");
  fprintf(f,"  -q, --quiet                 sets --debug-level=0\n");
}

static void printLicense(FILE * f, char * name)
{
  fprintf(f, "This program is free software; you can redistribute it and/or\n");
  fprintf(f, "modify it under the terms of the GNU General Public License\n");
  fprintf(f, "as published by the Free Software Foundation; either version 2\n");
  fprintf(f, "of the License, or (at your option) any later version.\n");
  fprintf(f, "\n");
  fprintf(f, "This program is distributed in the hope that it will be useful,\n");
  fprintf(f, "but WITHOUT ANY WARRANTY; without even the implied warranty of\n");
  fprintf(f, "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n");
  fprintf(f, "GNU General Public License for more details.\n");
  fprintf(f, "\n");
  fprintf(f, "You should have received a copy of the GNU General Public License\n");
  fprintf(f, "along with this program; if not, write to the Free Software\n");
  fprintf(f, "Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.\n");
  fprintf(f, "\n");
  fprintf(f, "See the COPYING file for license information.\n");
}

static void usage(poptContext optCon, FILE * f, char * name)
{
  poptPrintUsage(optCon, f, 0);
}

static void help(poptContext optCon, FILE * f, char * name)
{
  printTopDescription(f, name);
  poptPrintHelp(optCon, f, 0);
  printBottomDescription(f, name);
}

static void version(poptContext optCon, FILE * f, char * name)
{
  printTopDescription(f, name);
}

static void license(poptContext optCon, FILE * f, char * name)
{
  printTopDescription(f, name);
  printLicense(f, name);
}

void get_null_sa (LPSECURITY_ATTRIBUTES sap)
{
  static SECURITY_DESCRIPTOR *null_sdp = 0;
  static SECURITY_DESCRIPTOR sd;

  if (!sap)
    return;
  if (!null_sdp)
  {
    InitializeSecurityDescriptor (&sd, SECURITY_DESCRIPTOR_REVISION);
    SetSecurityDescriptorDacl (&sd, TRUE, 0, FALSE);
    null_sdp = &sd;
  }
  sap->nLength = sizeof *sap;
  sap->lpSecurityDescriptor = null_sdp;
  sap->bInheritHandle = TRUE;
}

