#!/bin/bash
# ======================================================================
# $Source: /home/cvs/cvsroot/src/cron/cron_diagnose.sh,v $
# $Date: 2004/12/13 17:09:13 $
# $Revision: 1.2 $
# $Name: rel14 $
# $State: Exp $
# $Author: corinna $
# ======================================================================

# ======================================================================
# This script checks that the various files, directories, and mount
# points needed by the cron daemon exist and have the proper settings,
# permissions, etc.  This script is based primarily on the requirements
# listed in the text file /usr/share/doc/Cygwin/cron.README.
# ======================================================================

# ======================================================================
# Routine: check_program
# Check to see that a specified program ($1) is installed and accessible
# by this script.  If it is not, then alert the user about which package
# ($2) should be installed to provide that program.
# ======================================================================
function check_program() {
	if ! type $1 > /dev/null 2>&1; then
		echo "The '$1' program is missing or not in your PATH."
		echo "This program is included in the '$2' package."
		echo "Please install this program and run this script again."
		return 1
	fi
} # === End of check_program() === #


# ======================================================================
# Routine: sanity_check
# Check for the set of programs that are used by this script.
# ======================================================================
function sanity_check() {
	# Check for programs that this script uses.
	check_program awk gawk || return
	check_program ls fileutils || return
	check_program grep grep || return
	check_program sed sed || return
	check_program id sh-utils || return
	check_program cut textutils || return
	check_program mount cygwin || return
	check_program cygcheck cygwin || return
} # === End of sanity_check() === #


# ======================================================================
# Routine: warning_for_etc_file
# Display a warning message for the user about overwriting the specified
# file in /etc.
# ======================================================================
function warning_for_etc_file() {
	echo
	echo "WARNING: The command above overwrites any existing /etc/$1."
	echo "You may want to preserve /etc/$1 before generating a new,"
	echo "one, and then compare your saved /etc/$1 file with the"
	echo "newly-generated one in case you need to restore other"
	echo "entries."
} # === warning_for_etc_file() === #


# ======================================================================
# Routine: check_system_in_group
# Check to see whether the first gid 18 maps to the SYSTEM sid
# in /etc/group.
# ======================================================================
function check_system_in_group() {
    if [ "$(sed -ne '/^[^:]*:[^:]*:18:/{s/^[^:]*:S-1-5-18:.*$/yes/p;q}' /etc/group)" != yes ]; then 
		echo "It appears that you do not have an entry mapping"
		echo "gid 18 to the SYSTEM sid in /etc/group."
		echo
		echo "Use the 'mkgroup' utility to generate it"
		echo "   mkgroup -l > /etc/group"
		echo "and then run this script again."
		warning_for_etc_file group
		return 1;
    fi
}  # === check_system_in_group() === #


# ======================================================================
# Routine: get_system_and_admins_gids
# Get the SYSTEM and ADMINs ids from /etc/group and /etc/passwd
# ======================================================================
function get_system_and_admins_ids() {
    ADMINSGID=$(sed -ne '/^[^:]*:S-1-5-32-544:.*:/{s/[^:]*:[^:]*:\([0-9]*\):.*$/\1/p;q}' /etc/group)
    SYSTEMGID=$(sed -ne '/^[^:]*:S-1-5-18:.*:/{s/[^:]*:[^:]*:\([0-9]*\):.*$/\1/p;q}' /etc/group)
    if [ -z "$ADMINSGID" -o -z "$SYSTEMGID" ]; then
		echo "It appears that you do not have entries for the"
		echo "SYSTEM and/or ADMINISTRATORS sids in /etc/group."
		echo
		echo "Use the 'mkgroup' utility to generate them"
		echo "   mkgroup -l > /etc/group"
		echo "and then run this script again."
		warning_for_etc_file group
		return 1;
    fi

    SYSTEMUID=$(sed -ne '/^[^:]*:[^:]*:[0-9]*:[0-9]*:[^:]*,S-1-5-18:.*:/{s/[^:]*:[^:]*:\([0-9]*\):.*$/\1/p;q}' /etc/passwd)
    if [ -z "$SYSTEMUID" ]; then
		echo "It appears that you do not have an entry for the"
		echo "SYSTEM sid in /etc/passwd."
		echo
		echo "Use the 'mkpasswd' utility to generate it"
		echo "   mkpasswd -l > /etc/passwd"
		echo "and then run this script again."
		warning_for_etc_file passwd
		return 1;
    fi
}  # === get_system_and_admins_ids() === #    


# ======================================================================
# Routine: check_passwd_and_group
# Check to see whether the user's password ID and group exist in the
# system /etc/passwd and /etc/group files, respectively.
# ======================================================================
function check_passwd_and_group() {
	if [ "$(id -gn)" = mkpasswd ]; then
		echo "It appears that you do not have an entry for your user ID"
		echo "in /etc/passwd.  If this check is incorrect, then re-run"
		echo "this script with the '-f' command-line option."
		echo
		echo "Otherwise, use the 'mkpasswd' utility to generate an"
		echo "entry for your User ID in the password file:"
		echo "   mkpasswd -l -u User_ID >> /etc/passwd"
		echo "or"
		echo "   mkpasswd -d -u User_ID >> /etc/passwd"
		echo "and then run this script again."
		return 1

	elif [ -n "$USERDOMAIN" ] && [ -n "$USERNAME" ]; then
		if ! grep -E -q -i "^$(id -un):.*U-$USERDOMAIN\\\\$USERNAME" /etc/passwd; then
			echo "It appears that you do not have an entry for:"
			echo "   $USERDOMAIN\\$USERNAME"
			echo "in /etc/passwd."
			echo
			echo "Use the 'mkpasswd' utility to generate an entry for"
			echo "your User ID in the password file:"
			echo "   mkpasswd -d -u User_ID >> /etc/passwd"
			echo "and then run this script again."
			return 1
		fi
	fi

	if [ "$(id -gn)" = mkgroup ]; then
		echo "It appears that you do not have an entry for your group ID"
		echo "in /etc/group.  If this check is incorrect, then re-run"
		echo "this script with the '-f' command-line option."
		echo
		echo "Otherwise, use the 'mkgroup' utility to generate an"
		echo "entry for your group ID in the password file:"
		echo "   mkgroup -l -g Group_id  >> /etc/group"
		echo "or"
		echo "   mkgroup -d -g Group_id >> /etc/group"
		echo "and then run this script again."
		return 1
	fi

	local fname=""
	for fname in /etc/passwd /etc/group; do
		if ! ls -ld $fname | grep -E -q '^-r..r..r..+? '; then
			echo "The permissions on the file $fname are not correct."
			echo "Please run 'chmod +r $fname', and then run this script again."
			return 1
		fi
	done
} # === End of check_passwd_and_group() === #


# ======================================================================
# Routine: check_dir
# Check to see that the specified directory ($1) exists.
# ======================================================================
function check_dir() {
	if [ ! -d $1 ]; then
		echo "Your computer does not appear to have a $1 directory."
		echo "Please investigate this problem, and then run this script again."
		return 1
	fi

	if ! ls -ld $1 | grep -E -q '^dr[-w]x.*'; then
		echo "The permissions on the directory $1 are not correct."
		echo "Please run 'chmod u+rx $1', and then run this script again."
		return 1
	fi
} # === End of check_dir() === #


# ======================================================================
# Routine: check_dir_perms
# Check to see that the specified directory ($1) exists and has the
# required permissions, as described in /usr/share/doc/Cygwin/cron.README.
# ======================================================================
function check_dir_perms() {
	check_dir $1 || return

	if ! ls -ld $1 | grep -F -q 'drwxrwxrwt'; then
		echo "The permissions on the directory $1 are not correct."
		echo "Please run 'chmod 1777 $1', and then run this script again."
		return 1
	fi
} # === End of check_dir_perms() === #


# ======================================================================
# Routine: check_var_run
# Check to see that SYSTEM or the Administrators group has write
# permission in the directory /var/run.  This permission is needed
# so that the cron.pid file can be created by the cron service.
# ======================================================================
function check_var_run() {
	local notify=0;

	# If SYSTEM is the owner of /var/run and does not have write
	# permission, then notify the user.
	if [ $(ls -dln /var/run | awk '{ print $3 }') -eq $SYSTEMUID ]; then
		if [ $(ls -dl /var/run | cut -b3) != w ]; then notify=1; fi
	# If 'Administrators' has group access to /var/run, but does not have
	# write permission, then notify the user.
	elif [ $(ls -dln /var/run | awk '{ print $4 }') -eq $ADMINSGID ]; then 
		if [ $(ls -dl /var/run | cut -b6) != w ]; then notify=1; fi
	# If 'everyone' / 'other' has write permission, then the permissions
	# MUST be sufficient for SYSTEM and administrators to write to it.
	elif [ $(ls -dln /var/run | cut -b9) != w ]; then notify=1; fi

	if [ $notify -eq 1 ]; then
		echo "The SYSTEM user or Administrators group needs to have";
		echo "write permission in the directory /var/run.";
		echo "Here are the permissions of this directory:";
		echo;
		ls -dl /var/run;
		echo;
		echo "Please change the user and/or group ownership and";
		echo "permissions of /var/run, and then run this script again.";
		return 1;
	fi
} # === End of check_var_run() === #


# ======================================================================
# Routine: check_sys_mount
# Check to see that the SYSTEM account has access to the specified
# directory.
# ======================================================================
function check_sys_mount() {
	local mnt_point=$1
	local dos_dir=$2

	echo -e "Checking mount point for $mnt_point. . .\c";
	if ! mount | grep -E -qe ".+ on $mnt_point .+system.+"; then
		echo;
		echo "The SYSTEM user cannot access the mount point ${mnt_point}."
		echo "Please run the following command to add a system mount point:"
		echo '   mount -f -s -b "[DOS path to Cygwin]'$dos_dir\" \"$mnt_point\"
		echo "where [DOS path to Cygwin] is something like c:/cygwin."
		echo
		echo "For more information, run 'mount -m' and 'mount -h'"
		echo
		echo "After adding this mount point, please re-run this script."
		return 1
	fi
	echo "done";
} # === End of check_sys_mount() === #


# ======================================================================
# Routine: check_cron_table
# Check for the existence of a crontab for the user, and check its
# permissions and ownership.
# ======================================================================
function check_cron_table() {
	local user_id=$(id -un)
	local cron_table=/var/cron/tabs/$user_id

	if [ ! -f $cron_table ]; then
		echo "Your computer does not appear to have a cron table for $user_id."
		echo "It is missing the file:"
		echo
		ls $cron_table
		echo
		echo "Please generate a cron table for $user_id using 'crontab -e',"
		echo "and run this script again."
		return 1
	fi

	if ! ls -l $cron_table | grep -F -q 'rw-r-----'; then
		echo "The file permissions of your cron table need to"
		echo "provide read/write access for $user_id."
		echo "The permissions of your cron table file are set to:"
		echo
		ls -l $cron_table
		echo
		echo "You can set the file permissions with:"
		echo "	chmod 640 $cron_table"
		echo "Please change your cron table's permissions, and run"
		echo "this script again."
		return 1
	fi

	if ! ls -ln $cron_table | awk '{ print $4 }' | grep -F -q "$SYSTEMGID"; then
		echo "The group membership of your cron table file should be SYSTEM,"
		echo "as documented in the file /usr/share/doc/Cygwin/cron.README."
		echo "Here is your cron table file:"
		echo
		ls -l $cron_table
		echo
		echo "You can change the group membership setting with:"
		echo "	 chgrp $SYSTEMGID $cron_table"
		echo "Please change your cron table's group membership, and"
		echo "run this script again."
		return 1
	fi
} # === End of check_cron_table() === #


function main() {
	echo -e "cron_diagnose.sh 1.9\n"

	sanity_check || return

	check_system_in_group || return
	get_system_and_admins_ids || return

	if [ "$1" != '-f' ]; then  # Allow the user to skip the password check.
		check_passwd_and_group || return
	fi

	# Check the integrity of the files in the 'cron' package:
	if cygcheck -c cron | grep -F -q 'Incomplete'; then
		echo "'cygcheck -c cron' reports that your cron installation"
		echo "is incomplete.  Please consider running 'setup.exe' and"
		echo "selecting 'Reinstall' from the install options, and then"
		echo "run this script again."
		return 1
	fi

	check_dir /etc/cron.d || return
	check_dir /var || return

	check_dir_perms /var/cron || return
	check_dir_perms /var/cron/tabs || return

	check_var_run || return

	check_sys_mount /usr/bin /bin || return
	check_sys_mount /usr/lib /lib || return
	check_sys_mount / / || return
	echo;

	check_cron_table || return

	echo "This script did not find any errors in your crontab setup."
	echo "If you are still unable to get cron to work, then try"
	echo "shutting down the cron service, uninstalling it,"
	echo "reinstalling it, and restarting it."
	echo
	echo "The following commands will do that:"
	echo "  $ cygrunsrv --stop cron"
	echo "  $ cygrunsrv --remove cron"
	echo "  $ cygrunsrv --install cron -p /usr/sbin/cron -a -D"
	echo "  $ cygrunsrv --start cron"
	echo 

	if [ -f /var/run/cron.pid ]; then
		echo "If the cron service does not start, try deleting the file"
		echo "/var/run/cron.pid and then repeating the commands above."
		echo
	fi

	echo "Also, examine the log file for cron, /var/log/cron.log, for"
	echo "information that it might give you about the problem cron is"
	echo "having."
	echo

	echo "If none of this fixes the problem, then report your problem"
	echo "to cygwin@cygwin.com.  Please include a copy of your crontab,"
	echo "('crontab -l') and the output of 'cygcheck -srv > cygcheck.txt'."
	echo
	echo "Please include the generated file 'cygcheck.txt' *as an attachment*,"
	echo "and NOT in the body of the mail message."
} # === End of main() === #

# Entry point:

# Set PATH to use the Cygwin programs
PATH=/usr/bin:/bin:$PATH

main $@
exit $?

# === End of $RCSfile: cron_diagnose.sh,v $ === #
